! $Id: ropp2bufr.f90 1959 2008-11-13 12:15:18Z frhl $

MODULE ropp2bufrmod

!****m* ropp2bufr/ropp2bufrmod *
!
! NAME
!   ropp2bufrmod - Module defining fixed values & subroutines/functions
!                  for the ropp2bufr main program
!
! USED BY
!   ropp2bufr
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

! Public fixed values for array sizes, etc

!   'Version' is the version number of this complete application program

  CHARACTER (LEN=*), PARAMETER :: Version = "V2.0 1-December-2008"  ! Program version

! GTS routing headers

  CHARACTER (LEN=3), PARAMETER :: TTA     = "IUT"  ! Binary/UAir/Satellite
  CHARACTER (LEN=2), PARAMETER :: ii      = "14"   ! Radio Occultation
  INTEGER,           PARAMETER :: RODescr = 310026 ! Table D descriptor for RO

  REAL,    PARAMETER :: MISSING  = -9999999.0 ! Missing data flag value
  INTEGER, PARAMETER :: BUFRunit = 20         ! BUFR file output stream
  INTEGER, PARAMETER :: Bulsunit = 21         ! Bulletin Seq. file I/O stream
  INTEGER, PARAMETER :: Input    = 1          ! Filemode: r   old
  INTEGER, PARAMETER :: Output   = 2          ! Filemode: w/r new

  INTEGER, PARAMETER :: NOhdrs   = 0          ! No GTS routings headers
  INTEGER, PARAMETER :: ONLYhdrs = 1          ! GTS Routing headers only
  INTEGER, PARAMETER :: X25hdrs  = 2          ! Headers plus X.25 support
  INTEGER, PARAMETER :: IPhdrs   = 3          ! Headers plus IP support

  INTEGER, PARAMETER :: ErrTim   =-1          ! Time threshold rejection
  INTEGER, PARAMETER :: ErrOK    = 0          ! No  error
  INTEGER, PARAMETER :: ErrIO    = 1          ! I/O error
  INTEGER, PARAMETER :: ErrMem   = 2          ! Memory allocation error

  LOGICAL            :: DEBUG    = .FALSE.    ! Debug mode

!--------------------------------------------------------------------
CONTAINS
!--------------------------------------------------------------------
SUBROUTINE GetOptions ( ROPPdsn,     & ! (out)
                        nfiles,      & ! (out)
                        BUFRdsn,     & ! (out)
                        BulSeqdsn,   & ! (out)
                        Thindsn,     & ! (out)
                        OrigICAO,    & ! (out)
                        OrigCentre,  & ! (out)
                        GTShdrType,  & ! (out)
                        RejTimeDiff, & ! (out)
                        CorrOnly,    & ! (out)
                        nomet,       & ! (out)
                        unordered )    ! (out)

!****s* ropp2bufr/GetOptions *
!
! NAME
!   GetOptions - Get command line information & options or set defaults
!
! ARGUMENTS
!   ROPPdsn      (out)  chr  ROPP input file name(s)
!   nfiles       (out)  int  No. of ROPP input files
!   BUFRdsn      (out)  chr  BUFR output file name
!   BulSeqdsn    (out)  chr  Bulletin Sequence file name
!   Thindsn      (out)  chr  Thinning control file name
!   OrigICAO     (out)  chr  Originator ICAO code
!   OrigCentre   (out)  int  Originator WMO code
!   GTShdrType   (out)  int  GTS header type code
!   RejTimeDiff  (out)  int  Rejection time threshold (minutes)
!   CorrOnly     (out)  log  L1+L2 skip flag
!   nomet        (out)  log  Met data skip flag
!   unordered    (out)  log  Disable profile ordering flag
!
! CALLS
!   IARGC
!   GETARG
!   CT001033
!
! CALLED BY
!   ropp2bufr
!
! INCLUDES
!   portability.fi  - system dependent settings (from BUFR package)
!                     to support IARGC(), GETARG() & EXIT()
!
! MODULES
!   DateTimeTypes   - Date & Time conversion definitions
!   ropp_tyoes
!
! SYNOPSIS
!    USE ropp2bufrmod
!    CHARACTER (LEN=100) :: roppdsn(100), bufrdsn, bulseqdsn
!    CHARACTER (LEN=4) :: origicao
!    INTEGER :: nfiles, origcentre, gtshdrtype, rejtimediff
!    LOGICAL :: corronly, nomet, unordered
!    CALL getOptions ( roppdsn, nfiles bufrdsn, bulseqdsn, thindsn, &
!                      origicao, origcentre, gtshdrtype, &
!                      rejtimediff, corronly, nomet, unordered )
!   On command line:
!   > ropp2bufr ropp_file [ropp_file...]
!                         [-o bufr_file] [-c orig_code] [-g[n]]
!                         [-s seq_file]  [-p thin_file] [-t time]
!                         [-u] [-l] [-m] [-h|?] [-v] [-d]
!
! DESCRIPTION
!   Provides a command line interface for the ROPP-to-BUFR
!   encoder application. See comments for main program ropp2bufr
!   for the command line details.
!
! SEE ALSO
!   ropp2bufr(1)
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

  USE CodeTables,    ONLY: CT001033
  USE DateTimeTypes, ONLY: nMinPerHour

! Include files

  INCLUDE 'portability.fi'

! Default Originating Centre (BUFR & ICAO) used if -c option
! is not given on command line. Possible codes for GPSRO include:
!  007/KWBC - Washington (USA)
!  074/EGRR - Exeter     (UK)
!  078/EDZW - Offenbach  (D)
!  094/EKMI - Copenhagen (DK)
! See BUFR code/flag table file (plain text version), code table 001033
! for the full list of Originating Centres.

  CHARACTER (LEN=*), PARAMETER :: DefOrigICAO   = "EGRR"  ! MetO Exeter
  INTEGER, PARAMETER           :: DefOrigCentre = 74      ! MetO Exeter

  CHARACTER (LEN=8), PARAMETER :: Defdsn = "ropp.nc"      ! Default input file name

  INTEGER, PARAMETER           :: DefRejTimeDiff = 1430   ! 23h50m in minutes

! Argument list parameters

  CHARACTER (LEN=*), INTENT(OUT) :: ROPPdsn(:)  ! input  ROPP file name(s)
  CHARACTER (LEN=*), INTENT(OUT) :: BUFRdsn     ! output BUFR file name
  CHARACTER (LEN=*), INTENT(OUT) :: BulSeqdsn   ! bulletin sequence file name
  CHARACTER (LEN=*), INTENT(OUT) :: Thindsn     ! thinning control file name
  CHARACTER (LEN=*), INTENT(OUT) :: OrigICAO    ! originating centre ICAO code
  INTEGER,           INTENT(OUT) :: nfiles      ! No. of ROPP input files
  INTEGER,           INTENT(OUT) :: OrigCentre  ! originating centre BUFR code
  INTEGER,           INTENT(OUT) :: GTShdrType  ! code for GTS header generation
  INTEGER,           INTENT(OUT) :: RejTimeDiff ! reject obs older than this
  LOGICAL,           INTENT(OUT) :: CorrOnly    ! .F. for L1+L2+C, .T. for C only
  LOGICAL,           INTENT(OUT) :: nomet       ! .F. for met data, .T. to skip
  LOGICAL,           INTENT(OUT) :: unordered   ! .T. to disable re-ordering of profiles

! Local variables

  INTEGER, PARAMETER :: descr = 001033        ! BUFR descriptor for originating
                                              !  centre code table
  CHARACTER (LEN=256) :: carg                 ! command line argument
  CHARACTER (LEN=30)  :: Centre               ! decoded ICAO name of orig. centre
  CHARACTER (LEN=256) :: ProgNam              ! program name

  INTEGER :: narg                             ! number of command line arguments
  INTEGER :: ia                               ! loop counter
  INTEGER :: ierr                             ! error status
  INTEGER :: hh, mm                           ! hours & minutes
  INTEGER :: oc                               ! originating centre

!-------------------------------------------------------------
! 1. Initialise
!-------------------------------------------------------------

  ROPPdsn(:)  = DefDsn
  nfiles      = 0
  BUFRdsn     = " "
  BulSeqdsn   = " "
  Thindsn     = "375"   ! to be interpreted as 'sample to no more than'
  OrigICAO    = " "
  OrigCentre  = DefOrigCentre
  GTShdrType  = NOhdrs
  RejTimeDiff = 0
  CorrOnly    = .FALSE.
  nomet       = .FALSE.
  unordered   = .FALSE.

  CALL GETARG ( 0, ProgNam )
  ia = LEN_TRIM(ProgNam)
  DO WHILE  ( ia > 0 .AND. &
              ProgNam(ia:ia) /= SysEndDirChr )
    ia = ia - 1
  END DO
  ProgNam = ProgNam(ia+1:)
  IF ( ProgNam == " " ) ProgNam = "ropp2bufr"

!-------------------------------------------------------------
! 2. Loop over all command line options.
!    If a switch has a trailing blank, then we need to get
!    the next string as it's argument.
!-------------------------------------------------------------

  ia   = 1
  narg = IARGC()

  DO WHILE ( ia <= narg )

    CALL GETARG ( ia, carg )
    IF ( carg(1:1) == "?" ) carg = "-h"

    IF ( carg(1:1) == "-" ) THEN   ! is this an option introducer?
                                   ! If so, which one?
      SELECT CASE (carg(2:2))

        CASE ("c","C")             ! Originating centre code
          carg(1:2) = "  "
          IF ( carg(3:) == " " ) THEN
             ia = ia + 1
             CALL GETARG ( ia, carg )
          END IF
          READ ( carg, *, IOSTAT=ierr ) oc
          IF ( ierr == 0 ) OrigCentre = oc

        CASE ("d","D")             ! debug/diagnostics wanted
          DEBUG = .TRUE.

        CASE ("g","G")             ! GTS headers wanted - any extra X.25 or IP?
          SELECT CASE (carg(3:3))
            CASE ("i","I" )
              GTShdrType = IPhdrs     ! headers + IP
            CASE ("x","X")
              GTShdrType = X25hdrs    ! headers + X.25
            CASE DEFAULT
              GTShdrType = ONLYhdrs   ! headers only
          END SELECT

        CASE ("l","L")             ! no L1/L2 (Corrected only)
          CorrOnly = .TRUE.

        CASE ("m","M")             ! no Met. (geophysical) data
          nomet = .TRUE.

        CASE ("o","O")             ! Output file name
          carg(1:2) = "  "
          IF ( carg(3:) == " " ) THEN
             ia = ia + 1
             CALL GETARG ( ia, carg )
          END IF
          BUFRdsn = ADJUSTL(carg)

        CASE ("p","P")             ! thinning control file name
          carg(1:2) = "  "
          IF ( carg(3:) == " " ) THEN
             ia = ia + 1
             CALL GETARG ( ia, carg )
          END IF
          Thindsn = ADJUSTL(carg)

        CASE ("s","S")             ! Bulletin Sequence No. file name
          carg(1:2) = "  "
          IF ( carg(3:) == " " ) THEN
             ia = ia + 1
             CALL GETARG ( ia, carg )
          END IF
          BulSeqdsn = ADJUSTL(carg)

        CASE ("t","T")             ! Reject time difference (hh:mm)
          carg(1:2) = "  "
          IF ( carg(3:) == " " ) THEN
             ia = ia + 1
             CALL GETARG ( ia, carg )
          END IF
          carg = ADJUSTL(carg)
          READ ( carg, "(BN,I2,1X,I2)", IOSTAT=ierr ) hh, mm
          IF ( ierr == 0 ) RejTimeDiff = hh * nMinPerHour + mm

        CASE ("u","U")             ! Profile ordering
          unordered = .TRUE.

        CASE ("h","H")              ! help
          WRITE ( *, * ) "Purpose:"
          WRITE ( *, * ) "  Encode one or more ROPP-format files to WMO BUFR."
          WRITE ( *, * ) "Usage:"
          WRITE ( *, * ) "> ",TRIM(ProgNam)," ropp_file [ropp_file...] [-o bufr_file]"
          WRITE ( *, * ) REPEAT(" ",LEN_TRIM(ProgNam)+13)//&
                            "[-c orig_code] [-g[n]] [-s seq_file]"
          WRITE ( *, * ) REPEAT(" ",LEN_TRIM(ProgNam)+13)//&
                            "[-p thin_file|maxsamp] [-t time]"
          WRITE ( *, * ) REPEAT(" ",LEN_TRIM(ProgNam)+13)//&
                            "[-u] [-l] [-d] [-m] [-h|?] [-v]"
          WRITE ( *, * ) "Input:"
          WRITE ( *, * ) "  Files must be in ROPP V1.0 (TEXT or netCDF)"
          WRITE ( *, * ) "   or CLIMAP V2.2 (TEXT) format."
          WRITE ( *, * ) "Options:"
          WRITE ( *, * ) "  -o  BUFR output file name"
          WRITE ( *, * ) "  -c  originating centre code value"
          WRITE ( *, * ) "  -g  GTS routing headers/trailers required"
          WRITE ( *, * ) "  -gx GTS routine headers preceded by 4 leading"
          WRITE ( *, * ) "      null bytes for GTS X.25 transmission"
          WRITE ( *, * ) "  -gi GTS headers preceded by 10-byte leading"
          WRITE ( *, * ) "      size/type for GTS IP (FTP) transmission"
          WRITE ( *, * ) "  -s  file containing last bulletin sequence number"
          WRITE ( *, * ) "      (updated on completion)"
          WRITE ( *, * ) "  -p  thinning control file name or max. no. samples"
          WRITE ( *, * ) "  -t  don't encode data older than 'time' ago (hh:mm)"
          WRITE ( *, * ) "  -u  leave profiles unordered (i.e. in original order)"
          WRITE ( *, * ) "  -l  specifies that L1+L2 data (Level 1b) are not encoded,"
          WRITE ( *, * ) "      only the ionospheric-corrected profile."
          WRITE ( *, * ) "  -m  specifies that met data (Level 2c/d) are not encoded"
          WRITE ( *, * ) "  -d  outputs additonal diagnostics to stdout"
          WRITE ( *, * ) "  -h or ?  writes this help (and does nothing else)"
          WRITE ( *, * ) "  -v  writes program version ID (and does nothing else)"
          WRITE ( *, * ) "Defaults:"
          WRITE ( *, * ) "  Input  file name        : ropp.nc"
          WRITE ( *, * ) "  Output file name        : from Occultation ID"
          WRITE ( *, * ) "  Originating centre code : 74 (Met Office, Exeter)"
          WRITE ( *, * ) "  GTS routing headers     : not generated"
          WRITE ( *, * ) "  Bulletin sequence nos.  : starts at 001"
          WRITE ( *, * ) "  Reject time difference  : 00:00 (no rejection on time)"
          WRITE ( *, * ) "   unless -g* option, when: 23:50 (assuming NRT on GTS)"
          WRITE ( *, * ) "  Thinning                : sample to <= 375 levels"
          WRITE ( *, * ) "  Re-ordering             : descending profiles re-ordered to ascending"
          WRITE ( *, * ) "Output:"
          WRITE ( *, * ) "  One output BUFR message"
          WRITE ( *, * ) "See ropp2bufr(1) for details."
          WRITE ( *, * ) " "
          CALL EXIT(ErrOK)

        CASE ("v","V")                 ! program version
          WRITE ( *, FMT="(A/)" ) TRIM(ProgNam)//": Version "// TRIM(Version)
          CALL EXIT(ErrOK)

        CASE DEFAULT                   ! unknown option
      END SELECT

    ELSE                               ! not an option - must be an input name
      nfiles = nfiles + 1
      ROPPdsn(nfiles) = carg
    END IF

    ia = ia + 1
  END DO                               ! argument loop

  IF ( nfiles == 0 ) nfiles = 1        ! No input files - try a default name

!-------------------------------------------------------------
! 3. Check originating centre code is valid
!-------------------------------------------------------------

  IF ( OrigCentre <=   0 .OR. &
       OrigCentre >= 255 ) THEN
    WRITE ( *, FMT="(A,I6,A,I3.3)" ) &
          "WARNING: Originating centre code ", OrigCentre, &
          " invalid - using default", DefOrigCentre
    OrigCentre = DefOrigCentre
  END IF

!-------------------------------------------------------------
! 4. If GTS header wanted, and not default, get 'cccc' code
!    from BUFR code table for specified orginating centre ID.
!    If not found, fall back to defaults.
!-------------------------------------------------------------

  IF ( GTShdrType /= NOhdrs .AND. &
       OrigCentre /= DefOrigCentre ) THEN
    CALL CT001033 ( OrigCentre, Centre, OrigICAO )
    IF ( OrigICAO == "UNKN" ) THEN
      WRITE ( *, FMT="(2(1X,A,I3.3,3A/))" )                    &
          "Warning: Originating centre ", OrigCentre,    " (", &
          TRIM(Centre), ") has no ICAO code",                  &
          "            - using default ", DefOrigCentre, " (", &
          DefOrigICAO, ")"
      OrigICAO   = DefOrigICAO
      OrigCentre = DefOrigCentre
    END IF

  ELSE
    OrigICAO = DefOrigICAO

  END IF

!-------------------------------------------------------------
! 5. Set default time rejection if GTS routing headers to be
!    generated, on the assumption that the output is for NRT
!    GTS distribution.
!-------------------------------------------------------------

  IF ( GTShdrType /= NOhdrs .AND. &
       RejTimeDiff == 0 ) RejTimeDiff = DefRejTimeDiff

END SUBROUTINE GetOptions
!---------------------------------------------------------------------------
SUBROUTINE BulSeqIO ( BulSeqdsn, & !(in)
                      BulSeqNo,  & !(inout)
                      inout )      !(in)

!****s* ropp2bufr/BulSeqNo *
!
! NAME
!   BulSeqNo - Read or save a bulletin sequence number
!
! ARGUMENTS
!   BulSeqdsn  (in)     chr  Bulletin sequence file
!   BulSeqNo   (inout)  int  Bulletin sequence number (001-999)
!   inout      (in)     int  Flag for input (<=1) or output (>=2)
!
! CALLED BY
!   ropp2bufr
!
! SYNOPSIS
!    CHARACTER (LEN=100) :: bulseqdsn
!    INTEGER :: bulseqno
!    CALL BulSeqIO ( bulseqdsn, bulseqno )
!
! DESCRIPTION
!   Reads (if inout <=1) or writes (if inout >=2) a bulletin
!   sequence number (which should be in the range 001-999) from/to
!   the given file. If the file name is blank, nothing happens.
!   Warning messages are written to stdout if any I/O error
!   occurs (the value of BulSeqNo is unchanged), but otherwise
!   the action is silent. The file name may include a path, and
!   should be accessible for read & write.
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

! Argument list parameters

  CHARACTER (LEN=*), INTENT(IN)    :: BulSeqdsn
  INTEGER,           INTENT(INOUT) :: BulSeqNo
  INTEGER,           INTENT(IN)    :: inout

! Local valitables

  INTEGER :: ierr

!-------------------------------------------------------------
! 1. Read previous bulletin sequence number...
!-------------------------------------------------------------

  IF ( inout <= 1 ) THEN

    IF ( BulSeqdsn /= " " ) THEN
      OPEN ( FILE=BulSeqdsn, &
             UNIT=BulsUnit,  &
           ACTION="READ",    &
           IOSTAT=ierr )
      IF ( ierr == 0 ) THEN
        READ ( UNIT=BulsUnit, &
                FMT=*,        &
             IOSTAT=ierr ) in
        IF ( ierr == 0 ) THEN
          BulSeqNo = in
        ELSE
          WRITE ( *, FMT="(A)" ) "WARNING: Invalid last Bulletin Sequence value"
        END IF
      ELSE
        WRITE ( *, FMT="(A)" ) "WARNING: Bulletin Sequence read file open error"
        WRITE ( *, FMT="(A)" ) "         "//TRIM(BulSeqdsn)
      END IF
      CLOSE ( UNIT=BulsUnit, &
            IOSTAT=ierr )
    END IF

!-------------------------------------------------------------
! 2. ...or save current bulletin sequence number
!-------------------------------------------------------------

  ELSE
    IF ( BulSeqdsn /= " " ) THEN
      OPEN ( FILE=BulSeqdsn, &
             UNIT=BulsUnit,  &
           ACTION="WRITE",   &
           IOSTAT=ierr )
      IF ( ierr == 0 ) THEN
        WRITE ( UNIT=BulsUnit, &
                 FMT="(I3.3)", &
              IOSTAT=ierr ) BulSeqNo
        IF ( ierr /= 0 ) &
           WRITE ( *, FMT="(A)" ) "WARNING: Failed to write Bulletin Sequence value"
      ELSE
        WRITE ( *, FMT="(A)" ) "WARNING: Bulletin Sequence save file open error"
        WRITE ( *, FMT="(A)" ) "         "//TRIM(BulSeqdsn)
      END IF
      CLOSE ( UNIT=BulsUnit, &
            IOSTAT=ierr )
    END IF
  END IF

END SUBROUTINE BulSeqIO
!----------------------------------------------------------------------------
SUBROUTINE ConvertROPPtoBUFR ( ROdata,     & ! (in)
                               CorrOnly,   & ! (in)
                               SubCentre,  & ! (out)
                               Values,     & ! (out)
                               Nvalues )     ! (out)
!
!****s* ropp2bufr/ConvertROPPtoBUFR *
!
! NAME
!   ConvertROPPtoBUFR - Convert ROPP data to BUFR specification
!
! ARGUMENTS
!   ROdata     (inout) dtyp  RO data - derived type
!   CorrOnly   (in)    log   Flag for corrected Level 1b profile only
!   SubCentre  (out)   int   Originating centre code value
!   Values     (out)   flt   Array(ne) of converted values for BUFR encoder
!   Nvalues    (out)   int   Total no. of values converted
!
! MODULES
!   -        - ROPP file I/O support
!
! CALLS
!   ConvertCodes
!
! CALLED BY
!   ropp2bufr
!
! SYNOPSIS
!    USE ropp_io_types
!    USE ropp2bufrmod
!    TYPE (ROprof) ROdata
!    INTEGER :: subcentre
!    REAL    :: Values(ne)
!    LOGICAL :: corronly
!    CALL ConvertROPPtoBUFR ( ROdata, corronly, &
!                             subcentre, values )
!   where
!    ne is the number of elements (data items for BUFR)
!
! DESCRIPTION
!   Converts RO data to BUFR units, etc, and returns converted
!   data as a plain array.
!   This procedure is mostly scaling and/or range changing
!   (e.g longitude from 0-360 to +/-180deg, hPa to Pa).
!   This routine also performs gross error checking, so that
!   if data is not valid (not within nominal range of BUFR bit width)
!   that data value is set "missing" in the output array.
!
! REFERENCES
!   1) ROPP interface file format
!      SAF/GRAS/METO/FMT/ROPP/001
!   2) WMO FM94 (BUFR) Specification for GRAS SAF Processed Radio
!      Occultation Data. SAF/GRAS/UKMO/FMT/BUFR/01
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

! Modules

  USE ropp_io_types, ONLY: ROprof, &
                           PCD_occultation

  IMPLICIT NONE

! Fixed parameters

  INTEGER, PARAMETER :: MISSING = -9999999   ! Missing data flag value

  INTEGER, PARAMETER :: ProdType = 2         ! Product type (limb sounding)
  INTEGER, PARAMETER :: FOstats  = 13        ! First-order statistics (rms)

  REAL,    PARAMETER :: FreqL1   = 1.5E9     ! L1 frequency: 1.5GHz
  REAL,    PARAMETER :: FreqL2   = 1.2E9     ! L1 frequency: 1.2GHz
  REAL,    PARAMETER :: FreqLc   = 0.0       ! Corrected frequency (dummy)

  CHARACTER (LEN=*), PARAMETER :: numeric = "0123456789." ! valid for numerics

! Argument list parameters

  TYPE (ROprof), INTENT(INOUT):: ROdata
  LOGICAL,       INTENT(IN)   :: CorrOnly
  INTEGER,       INTENT(OUT)  :: SubCentre
  REAL,          INTENT(OUT)  :: Values(:)
  INTEGER,       INTENT(OUT)  :: Nvalues

! Local parameters

  CHARACTER (LEN=10) :: strnum ! temporary strings for numeric values
  INTEGER :: Gclass     ! GNSS class value
  INTEGER :: Gcode      ! GNSS PRN
  INTEGER :: Lcode      ! LEO  code value
  INTEGER :: Icode      ! Instrument code value
  INTEGER :: Ocode      ! Origin. centre code value
  INTEGER :: Scode      ! Sub-centre code value
  INTEGER :: Bcode      ! B/G generator code value
  INTEGER :: PCD        ! PCD bit flags (16-bit)
  INTEGER :: in         ! loop counter for profile arrays
  INTEGER :: IE         ! index offset to Values element
  INTEGER :: ierr       ! I/O error code
  REAL    :: SWver      ! Softwre version number

!-------------------------------------------------------------
! 1. Initialise
!------------------------------------------------------------

  Values(:) = MISSING

!-------------------------------------------------------------
! 2. Convert ROPP character codes to BUFR numeric codes.
!-------------------------------------------------------------

  CALL ConvertCodes ( ROdata,        &
                      Gclass, Gcode, &
                      Lcode,  Icode, &
                      Ocode,  Scode, &
                      Bcode,         &
                      1 )
  SubCentre = Scode

!-------------------------------------------------------------
! 3. Satellite data introducer
!-------------------------------------------------------------

  Values(1) = Lcode                                    ! LEO ID
  Values(2) = Icode                                    ! RO Instrument
  IF ( BTEST(ROdata%PCD,PCD_occultation) ) THEN
    Values(3) = Bcode                                  ! B/g gen.centre
  ELSE
    Values(3) = Ocode                                  ! Proc.centre
  END IF
  Values(4) = ProdType                                 ! Product type

  strnum = ROdata%Software_Version(1:10)
  DO in = 1, LEN_TRIM(strnum)
    IF ( INDEX ( numeric, strnum(in:in) ) == 0 ) strnum(in:in) = " "
  END DO
  READ ( strnum, FMT=*, IOSTAT=ierr) SWver
  IF ( ierr /= 0 ) SWver = -9.999
  Values(5) = SWver * 1E3                              ! Software version
  IF ( Values(5) <     0.0 .OR.  &
       Values(5) > 16382.0 ) &
       Values(5) = MISSING

! Date/time of start of occultation (or background profile)

  Values(6)  = 17.                                     ! Time.sig (start)
  Values(7)  = ROdata%DTocc%Year                       ! Year
  Values(8)  = ROdata%DTocc%Month                      ! Month
  Values(9)  = ROdata%DTocc%Day                        ! Day
  Values(10) = ROdata%DTocc%Hour                       ! Hour
  Values(11) = ROdata%DTocc%Minute                     ! Minute
  Values(12) = ROdata%DTocc%Second &                   ! Seconds & MSecs
             + ROdata%DTocc%MSec * 1E-3
  IF ( Values(12) <  0.000 .OR. &
       Values(12) > 59.999 )    &
       Values(12) = MISSING

! Summary quality information

  PCD = 0
  DO in = 0, 15
    IF ( BTEST(ROdata%PCD, in) ) PCD = IBSET(PCD, 15-in) ! only use 1st 16 bits in swapped bit order
  END DO
  Values(13) = REAL(PCD)                               ! PCD
  IF ( Values(13) <     0.0 .OR. &
       Values(13) > 65534.0 )    &
       Values(13) = MISSING

  Values(14) = REAL(ROdata%Overall_Qual)               ! Percent confidence
  IF ( Values(14) <   0.0  .OR. &
       Values(14) > 100.0 )     &
       Values(14) = MISSING

! LEO & GNSS POD

  IF ( ROdata%Lev1a%Npoints > 0 ) THEN
    Values(15) = REAL(ROdata%Lev1a%R_LEO(1,1))         ! LEO X posn (m)
    IF ( ABS(Values(15)) > 10737418.23 ) &
             Values(15) = MISSING
    Values(16) = REAL(ROdata%Lev1a%R_LEO(1,2))         ! LEO Y posn (m)
    IF ( ABS(Values(16)) > 10737418.23 ) &
             Values(16) = MISSING
    Values(17) = REAL(ROdata%Lev1a%R_LEO(1,3))         ! LEO Z posn (m)
    IF ( ABS(Values(17)) > 10737418.23 ) &
             Values(17) = MISSING
    IF ( ABS(Values(15)) < 1.0 .AND. &
         ABS(Values(16)) < 1.0 .AND. &
         ABS(Values(17)) < 1.0 )     &
             Values(15:17) = MISSING
    Values(18) = REAL(ROdata%Lev1a%V_LEO(1,1))         ! LEO X vely (m/s)
    IF ( ABS(Values(18)) > 10737.41823 ) &
             Values(18) = MISSING
    Values(19) = REAL(ROdata%Lev1a%V_LEO(1,2))         ! LEO Y vely (m/s)
    IF ( ABS(Values(19)) > 10737.41823 ) &
             Values(19) = MISSING
    Values(20) = REAL(ROdata%Lev1a%V_LEO(1,3))         ! LEO Z vely (m/s)
    IF ( ABS(Values(20)) > 10737.41823 ) &
             Values(20) = MISSING
    IF ( ABS(Values(18)) < 1.0 .AND. &
         ABS(Values(19)) < 1.0 .AND. &
         ABS(Values(20)) < 1.0 )     &
             Values(18:20) = MISSING
  END IF

  Values(21) = Gclass                                  ! GNSS class
  Values(22) = Gcode                                   ! GNSS PRN

  IF ( ROdata%Lev1a%Npoints > 0 ) THEN
    Values(23) = REAL(ROdata%Lev1a%R_GNS(1,1))         ! GNSS X posn (m)
    IF ( ABS(Values(23)) > 107374182.4 ) &
             Values(23) = MISSING
    Values(24) = REAL(ROdata%Lev1a%R_GNS(1,2))         ! GNSS Y posn (m)
    IF ( ABS(Values(24)) > 107374182.4 ) &
             Values(24) = MISSING
    Values(25) = REAL(ROdata%Lev1a%R_GNS(1,3))         ! GNSS Z posn (m)
    IF ( ABS(Values(25)) > 107374182.4 ) &
             Values(25) = MISSING
    IF ( ABS(Values(23)) < 1.0 .AND. &
         ABS(Values(24)) < 1.0 .AND. &
         ABS(Values(25)) < 1.0 )     &
             Values(23:25) = MISSING

    Values(26) = REAL(ROdata%Lev1a%V_GNS(1,1))         ! GNSS X vely (m/s)
    IF ( ABS(Values(26)) > 10737.41824 ) &
             Values(26) = MISSING
    Values(27) = REAL(ROdata%Lev1a%V_GNS(1,2))         ! GNSS Y vely (m/s)
    IF ( ABS(Values(27)) > 10737.41824 ) &
             Values(27) = MISSING
    Values(28) = REAL(ROdata%Lev1a%V_GNS(1,3))         ! GNSS Z vely (m/s)
    IF ( ABS(Values(28)) > 10737.41824 ) &
             Values(28) = MISSING
    IF ( ABS(Values(26)) < 1.0 .AND. &
         ABS(Values(27)) < 1.0 .AND. &
         ABS(Values(28)) < 1.0 )     &
             Values(26:28) = MISSING
  END IF

! Local Earth parameters

  Values(29) = REAL(ROdata%GeoRef%Time_Offset)         ! Time/start (s)
  IF ( Values(29) <    0.0 .OR. &
       Values(29) >  240.0 )    &
       Values(29) = MISSING

  Values(30) = REAL(ROdata%GeoRef%Lat)                 ! Latitude (deg)
  IF ( ABS(Values(30)) > 90.0 ) &
       Values(30) = MISSING

  Values(31) = REAL(ROdata%GeoRef%Lon)                 ! Longitude (deg)
  IF ( Values(31) > 180.0 ) &
       Values(31) = Values(31) - 360.0
  IF ( ABS(Values(31)) > 180.0 ) &
       Values(31) = MISSING

  Values(32) = REAL(ROdata%GeoRef%r_CoC(1))            ! CofC X (m)
  IF ( ABS(Values(32)) > 1000000.0 ) &
       Values(32) = MISSING

  Values(33) = REAL(ROdata%GeoRef%r_CoC(2))            ! CofC Y (m)
  IF ( ABS(Values(33)) > 1000000.0 ) &
       Values(33) = MISSING

  Values(34) = REAL(ROdata%GeoRef%r_CoC(3))            ! CofC Z (m)
  IF ( ABS(Values(34)) > 1000000.0 ) &
       Values(34) = MISSING

  Values(35) = REAL(ROdata%GeoRef%RoC)                 ! Radius value (m)
  IF ( Values(35) < 6200000.0 .OR. &
       Values(35) > 6600000.0 )    &
       Values(35) = MISSING

  Values(36) = REAL(ROdata%GeoRef%Azimuth)             ! Line of sight bearing (degT)
  IF ( Values(36) <    0.0 .OR. &
       Values(36) >= 360.0 )    &
       Values(36) = MISSING

  Values(37) = REAL(ROdata%GeoRef%Undulation)          ! Geoid undulation (m)
  IF ( ABS(Values(37)) > 163.82 ) &
       Values(37) = MISSING

  IE = 37

!-------------------------------------------------------------
! 4. Level 1b data (bending angle profile)
!-------------------------------------------------------------

  Values(IE+1) = ROdata%Lev1b%Npoints                  ! Replication factor

  DO in = 1, ROdata%Lev1b%Npoints

! Coordinates

    Values(IE+2) = REAL(ROdata%Lev1b%Lat_tp(in))       ! Latitude (deg)
    IF ( ABS(Values(IE+2)) > 90.0 ) &
         Values(IE+2) = MISSING

    Values(IE+3) = REAL(ROdata%Lev1b%Lon_tp(in))       ! Longitude (deg)
    IF ( Values(IE+3) > 180.0 ) &
         Values(IE+3) = Values(IE+3) - 360.0
    IF ( ABS(Values(IE+3)) > 180.0 ) &
         Values(IE+3) = MISSING

    Values(IE+4) = REAL(ROdata%Lev1b%Azimuth_tp(in))   ! Line of sight bearing (degT)
    IF ( Values(IE+4) <    0.0 .OR. &
         Values(IE+4) >= 360.0 )    &
         Values(IE+4) = MISSING

! Include L1+L2 or skip them?

    IF ( CorrOnly ) THEN
      Values(IE+5) = 1                                 ! Replication factor
      IE = IE - 12
    ELSE
      Values(IE+5) = 3

! L1 data

      Values(IE+6) = FreqL1                            ! L1=1.5Ghz

      Values(IE+7) = REAL(ROdata%Lev1b%Impact_L1(in))  ! Impact parameter (m)
      IF ( Values(IE+7) < 6200000.0 .OR. &
           Values(IE+7) > 6600000.0 )    &
           Values(IE+7) = MISSING

      Values(IE+8) = REAL(ROdata%Lev1b%BAngle_L1(in))  ! B/angle (rad)
      IF ( Values(IE+8) < -1.0E-3 .OR. &
           Values(IE+8) >  8.288E-2 )    &
           Values(IE+8) = MISSING

      Values(IE+9) = FOstats                           ! 1st order stats (rms)

      Values(IE+10) = REAL(ROdata%Lev1b%BAngle_L1_Sigma(in)) ! B/angle error (rad)
      IF ( Values(IE+10) < 0.0 .OR. &
           Values(IE+10) > 1.0E-2 ) &
           Values(IE+10) = MISSING
      Values(IE+10) = MIN ( Values(IE+10), 0.00948 )   ! allow for BUFR offset

      Values(IE+11) = MISSING                          ! 1st order stats (off)

! L2 data

      Values(IE+12) = FreqL2                           ! L2=1.2Ghz

      Values(IE+13) = REAL(ROdata%Lev1b%Impact_L2(in)) ! Impact parameter (m)
      IF ( Values(IE+13) < 6200000.0 .OR. &
           Values(IE+13) > 6600000.0 )    &
           Values(IE+13) = MISSING

      Values(IE+14) = REAL(ROdata%Lev1b%BAngle_L2(in)) ! B/angle (rad)
      IF ( Values(IE+14) < -1.0E-3 .OR. &
           Values(IE+14) >  8.288E-2 )    &
           Values(IE+14) = MISSING

      Values(IE+15) = FOstats                          ! 1st order stats (rms)

      Values(IE+16) = REAL(ROdata%Lev1b%BAngle_L2_Sigma(in)) ! B/angle error (rad)
      IF ( Values(IE+16) < 0.0 .OR. &
           Values(IE+16) > 1.0E-2 ) &
           Values(IE+16) = MISSING
      Values(IE+16) = MIN ( Values(IE+16), 0.00948 )   ! allow for BUFR offset

      Values(IE+17) = MISSING                          ! 1st order stats (off)
   END IF

! Corrected bending angle (always encoded)

    Values(IE+18) = FreqLc                             ! corrected

    Values(IE+19) = REAL(ROdata%Lev1b%Impact(in))      ! Impact parameter (m)
    IF ( Values(IE+19) < 6200000.0 .OR. &
         Values(IE+19) > 6600000.0 )    &
         Values(IE+19) = MISSING

    Values(IE+20) = REAL(ROdata%Lev1b%BAngle(in))      ! B/Ang (rad)
    IF ( Values(IE+20) < -1.0E-3 .OR. &
         Values(IE+20) >  8.288E-2 )    &
         Values(IE+20) = MISSING

    Values(IE+21) = FOstats                            ! 1st order stats (rms)

    Values(IE+22) = REAL(ROdata%Lev1b%BAngle_Sigma(in)) ! Error in B/Ang (rad)
    IF ( Values(IE+22) < 0.0 .OR. &
         Values(IE+22) > 1.0E-2 ) &
         Values(IE+22) = MISSING
    Values(IE+22) = MIN ( Values(IE+22), 0.00948 )     ! allow for BUFR offset

    Values(IE+23) = MISSING                            ! 1st order stats (off)

    Values(IE+24) = REAL(ROdata%Lev1b%Bangle_Qual(in)) ! Percent confidence
    IF ( Values(IE+24) <   0.0  .OR. &
         Values(IE+24) > 100.0 )     &
         Values(IE+24) = MISSING

    IE = IE + 23
  END DO
  IE = IE + 1

!-------------------------------------------------------------
! 5. Level 2a data (derived refractivity profile)
!-------------------------------------------------------------

  Values(IE+1) = ROdata%Lev2a%Npoints                  ! Replication factor

  DO in = 1, ROdata%Lev2a%Npoints

    Values(IE+2) = REAL(ROdata%Lev2a%Alt_Refrac(in))   ! Height amsl (m)
    IF ( Values(IE+2) <  -1000.0 .OR. &
         Values(IE+2) > 100000.0 )    &
         Values(IE+2) = MISSING

    Values(IE+3) = REAL(ROdata%Lev2a%Refrac(in))       ! Refrac (N-units)
    IF ( Values(IE+3) <   0.0 .OR. &
         Values(IE+3) > 524.0 )    &
         Values(IE+3) = MISSING

    Values(IE+4) = FOstats                             ! 1st order stats (rms)

    Values(IE+5) = REAL(ROdata%Lev2a%Refrac_Sigma(in)) ! Refrac error (N-units)
    IF ( Values(IE+5) <  0.0 .OR. &
         Values(IE+5) > 16.382 )  &
         Values(IE+5) = MISSING

    Values(IE+6) = MISSING                             ! 1st order stats (off)

    Values(IE+7) = REAL(ROdata%Lev2a%Refrac_Qual(in))  ! Percent confidence
    IF ( Values(IE+7) <   0.0  .OR. &
         Values(IE+7) > 100.0 )     &
         Values(IE+7) = MISSING

   IE = IE + 6
  END DO
  IE = IE + 1

!-------------------------------------------------------------
! 6. Level 2b data (retrieved P,T,q profile)
!-------------------------------------------------------------

  Values(IE+1) = ROdata%Lev2b%Npoints                  ! Replication factor

  DO in = 1, ROdata%Lev2b%Npoints

    Values(IE+2) = REAL(ROdata%Lev2b%Geop(in))         ! Geopot ht (gpm)
    IF ( Values(IE+2) <  -1000.0 .OR. &
         Values(IE+2) > 100000.0 )    &
         Values(IE+2) = MISSING

    Values(IE+3) = REAL(ROdata%Lev2b%Press(in)) * 1E2  ! Pressure (Pa)
    IF ( Values(IE+3) <=      0.0 .OR. &               ! Min. 0.1hPa
         Values(IE+3) >  150000.0 )    &
         Values(IE+3) = MISSING

    Values(IE+4) = REAL(ROdata%Lev2b%Temp(in))         ! Temperature (K)
    IF ( Values(IE+4) < 150.0 .OR. &
         Values(IE+4) > 350.0 )    &
         Values(IE+4) = MISSING

    Values(IE+5) = REAL(ROdata%Lev2b%SHum(in)) * 1E-3  ! Spec/humidity (Kg/Kg)
    IF ( Values(IE+5) <  0.0 .OR. &
         Values(IE+5) >  0.16 )   &
         Values(IE+5) = MISSING

    Values(IE+6) = FOstats                             ! 1st order stats (rms)

    Values(IE+7) = REAL(ROdata%Lev2b%Press_Sigma(in)) * 1E2  ! Pressure error (Pa)
    IF ( Values(IE+7) <   0.0 .OR. &
         Values(IE+7) > 620.0 )    &
         Values(IE+7) = MISSING

    Values(IE+8) = REAL(ROdata%Lev2b%Temp_Sigma(in))   ! Temperature error (K)
    IF ( Values(IE+8) < 0.0 .OR. &
         Values(IE+8) > 6.2 )    &
         Values(IE+8) = MISSING

    Values(IE+9) = REAL(ROdata%Lev2b%SHum_Sigma(in)) * 1E-3  ! S/Hum error (Kg/Kg)
    IF ( Values(IE+9) < 0.0 .OR. &
         Values(IE+9) > 0.0051 ) &
         Values(IE+9) = MISSING

    Values(IE+10) = MISSING                            ! 1st order stats (off)

    Values(IE+11) = REAL(ROdata%Lev2b%Meteo_Qual(in))  ! Percent confidence
    IF ( Values(IE+11) <   0.0  .OR. &
         Values(IE+11) > 100.0 )     &
         Values(IE+11) = MISSING

   IE = IE + 10
  END DO
  IE = IE + 1

!-------------------------------------------------------------
! 7. Level 2c data (retrieved surface params)
!-------------------------------------------------------------

  Values(IE+1) = 0                                     ! Vertical sig. (surf)

  VALUES(IE+2) = REAL(ROdata%Lev2c%Geop_Sfc)           ! Geoptot.Ht. (of surf) (gpm)
  IF ( Values(IE+2) < -1000.0 .OR. &
       Values(IE+2) > 10000.0 )    &
       Values(IE+2) = MISSING

  Values(IE+3) = REAL(ROdata%Lev2c%Press_Sfc) * 1E2    ! Surface pressure (Pa)
  IF ( Values(IE+3) <      0.0 .OR. &
       Values(IE+3) > 150000.0 )    &
       Values(IE+3) = MISSING

  Values(IE+4) = FOstats                               ! 1st order stats (rms)

  Values(IE+5) = REAL(ROdata%Lev2c%Press_Sfc_Sigma) * 1E2 ! S/press error (Pa)
  IF ( Values(IE+5) <   0.0 .OR. &
       Values(IE+5) > 620.0 )    &
       Values(IE+5) = MISSING

  Values(IE+6) = MISSING                               ! 1st order stats (off)

  Values(IE+7) = REAL(ROdata%Lev2c%Press_Sfc_Qual)     ! Percent confidence
  IF ( Values(IE+7) <   0.0  .OR. &
       Values(IE+7) > 100.0 )     &
       Values(IE+7) = MISSING

  Nvalues = IE + 7                                     ! Total no. of values

END SUBROUTINE ConvertROPPtoBUFR
!---------------------------------------------------------------------
SUBROUTINE ConvertCodes ( ROdata, & ! (inout)
                          Gclass, & ! (inout)
                          Gcode,  & ! (inout)
                          Lcode,  & ! (inout)
                          Icode,  & ! (inout)
                          Ocode,  & ! (inout)
                          Scode,  & ! (inout)
                          Bcode,  & ! (inout)
                          ind )     ! (in)

!****s* bufr2ropp/ConvertCodes *
!
! NAME
!   ConvertCodes - Convert header codes between ROPP and BUFR
!
! ARGUMENTS
!   ROdata  (inout)  dtyp  RO data structure
!   Gclass  (inout)  int   GNSS code (Satellite Class)
!   Gcode   (inout)  int   GNSS PRN  (Platform Tx ID)
!   Lcode   (inout)  int   LEO  code (Satellite ID)
!   Icode   (inout)  int   Instrument code (Instrument ID)
!   Ocode   (inout)  int   Originating (processing) Centre code
!   Scode   (inout)  int   Sub-centre code
!   Bcode   (inout)  int   Background generating centre code
!   ind     (in)     int   ROPP-->BUFR if >0, else BUFR-->ROPP
!
! MODULES
!   ropp_io             - ROPP file I/O support
!
! DEPENDENCIES
!   MetDB BUFR package  - BUFR kernel routines
!
! CALLS
!   BUFRPATH
!
! CALLED BY
!   ConvertBUFRtoROPP
!   ConvertROPPtoBUFR
!
! NAMELISTS
!   roppbufrcodes.nl    -  in path BUFR_LIBRARY
!
! SYNOPSIS
!    USE ropp_io_types
!    TYPE (roprof) rodata
!    INTEGER :: gclass,gcode,lcode,icode,ocode,scode,bcode,ind
!    ind = 1 ! to convert ROPP-->BUFR, ind =-1 for BUFR-->ROPP
!    CALL convertcodes(rodata,&
!                       glass,gcode,lcode,icode,ocode,scode,bcode,&
!                       ind)
!
! DESCRIPTION
!   Converts from character-based codes (as defined for ROPP)
!   to numeric codes suitable for BUFR encoding, if ind>0, else
!   vice-versa.
!   The code conversion is driven by a set of look-up tables, which
!   are read from a NAMELIST file 'roppbufrcodes.nl' which is
!   expected in the directory path defined by environment variable
!   BUFR_LIBRARY. If this file cannot be opened, a warning is issued
!   and an in-built default set of tables is used instead.
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

! Modules

  USE ropp_io_types

  IMPLICIT NONE

! Fixed values

  INTEGER, PARAMETER :: MISSING = -9999999      ! Missing data flag value

!  NB: no. of elements given in NAMELIST file parameters must not
!  exceed these values - increase values below if necessary.

  INTEGER, PARAMETER :: ntx     = 5             ! Max. no. of GNSS Tx types
  INTEGER, PARAMETER :: nrx     = 20            ! Max. no. of LEO  Rx types
  INTEGER, PARAMETER :: noc     = 10            ! Max. no. of orig. centre types
  INTEGER, PARAMETER :: nbg     = 10            ! Max. no. of b/g centre types

  CHARACTER (LEN=*), PARAMETER :: NLdsn  = "roppbufrcodes.nl" ! NAMELIST filel name
  INTEGER,           PARAMETER :: NLunit = 1    ! NAMELIST file unit no.

! Argument list parameters

  TYPE ( ROprof ), INTENT(INOUT) :: Rodata      ! ROPP data structure
  INTEGER,         INTENT(INOUT) :: Gclass      ! GNSS class value
  INTEGER,         INTENT(INOUT) :: Gcode       ! GNSS PRN
  INTEGER,         INTENT(INOUT) :: Lcode       ! LEO  code value
  INTEGER,         INTENT(INOUT) :: Icode       ! Instrument code value
  INTEGER,         INTENT(INOUT) :: Ocode       ! Origin. centre code value
  INTEGER,         INTENT(INOUT) :: Scode       ! Sub-centre code value
  INTEGER,         INTENT(INOUT) :: Bcode       ! B/G generator code value
  INTEGER,         INTENT(IN)    :: ind         ! RO->code if >1 else code->RO

! Define arrays for chararacter (ROPP) & numeric (BUFR code) lists.
! Set some defaults in case the NAMELISTs can't be read. NAMELIST
! values will overwrite these defaults. Include some dummy spares so
! that extra ones can be defined in the NAMELIST _without_ having to
! change the array sizes (up to the max. values) and rebuilding
! the program.

  CHARACTER (LEN=1),  DIMENSION(ntx) :: GNSlist = &
                      (/  "U",    "G", "R", "E",   "U"   /)
  INTEGER,            DIMENSION(ntx) :: GNScode = &
                      (/ MISSING, 401, 402, 403, MISSING /)

  CHARACTER (LEN=4),  DIMENSION(nrx) :: LEOlist = &
                      (/ "UNKN", "OERS", "CHMP",         &
                                 "SUNS", "SACC",         &
                                 "GRAA", "GRAB",         &
                                 "CO01", "CO02", "CO03", &
                                 "CO04", "CO05", "CO06", &
                                 "META", "METB", "METC", &
                                 "TSRX",                 &
                                 "UNKN", "UNKN", "UNKN" /)
  INTEGER,            DIMENSION(nrx) :: LEOcode =  &
                      (/ MISSING,  040,    041,         &
                                   800,    820,         &
                                   722,    723,         &
                                   740,    741,    742, &
                                   743,    744,    745, &
                                   004,    003,    005, &
                                   042,                 &
                                   MISSING, MISSING, MISSING /)
  INTEGER,            DIMENSION(nrx) :: Inscode =  &
                      (/ MISSING,  102,    102,         &
                                   102,    102,         &
                                   102,    102,         &
                                   102,    102,    102, &
                                   102,    102,    102, &
                                   202,    202,    202, &
                                   103,                 &
                                   MISSING, MISSING, MISSING /)

! List of (BUFR) Originating Centre IDs & their BUFR codes
! (Code Table 001033, CCT C-1, or 001035, CCT C-11)
! The (Processing) Sub-centre code should be valid for the
! associated Originating Centre code (for which Sub-Centre is 0).
! (Code Table 001034, CCT C-12)
!
  CHARACTER (LEN=8), DIMENSION(noc) :: ORGlist = &
                      (/ "UNKNOWN ", "DMI     ", "GFZ     ",  &
                                     "METO    ", "UCAR    ",  &
                                     "NESDIS  ", "EUMETSAT", &
                         "UNKNOWN ", "UNKNOWN ", "UNKNOWN " /)
  INTEGER,            DIMENSION(noc) :: ORGcode = &
                      (/ MISSING,    094,       078,  &
                                     074,       060,  &
                                     160,       254, &
                         MISSING,    MISSING,   MISSING /)
  INTEGER,            DIMENSION(noc) :: Subcode = &
                      (/     000,    000,        173,  &
                                     000,        000,  &
                                     000,        000,  &
                             000,    000,        000 /)
  CHARACTER (LEN=35), DIMENSION(noc) :: ORGname = &
                                    (/ "                                   ", &
                                       "(GRAS SAF)                         ", &
                                       "GeoForschungsZentrum Potsdam       ", &
                                       "Met Office Exeter                  ", &
                                       "Boulder                            ", &
                                       "Washington                         ", &
                                       "Darmstatdt                         ", &
                                       "                                   ", &
                                       "                                   ", &
                                       "                                   " /)

  CHARACTER (LEN=20), DIMENSION(nbg) :: BGDlist = &
                      (/ "UNKNOWN", "ECMWF  ", "DMI    ", &
                                    "METO   ", "NCEP   ", &
                                    "NONE   ",            &
                         "UNKNOWN", "UNKNOWN", "UNKNOWN", "UNKNOWN" /)
  INTEGER,            DIMENSION(nbg) :: BGDcode = &
                       (/ MISSING,    98,     94,     74,      7,   &
                          MISSING, MISSING, MISSING, MISSING, MISSING /)

! Local variables

  CHARACTER (LEN=235) :: dir = " "       ! Translated BUFR directory (path)
  CHARACTER (LEN=255) :: FileSpec        ! Full sequence file name
  INTEGER :: i, j                        ! loop counter/indices
  INTEGER :: ierr                        ! I/O error

! Namelist parameters

  NAMELIST /GNScodes/ GNSlist, GNScode
  NAMELIST /LEOcodes/ LEOlist, LEOcode, Inscode
  NAMELIST /ORGcodes/ ORGlist, ORGcode, Subcode
  NAMELIST /BGDcodes/ BGDlist, BGDcode

!---------------------------------------------------
! 1. Open codes NAMELIST file & read lists
!---------------------------------------------------

  CALL BUFRPATH ( dir, ierr )
  FileSpec = ADJUSTL(TRIM(dir)//NLdsn)
  OPEN ( UNIT=NLunit, FILE=FileSpec, ACTION="READ", IOSTAT=ierr )
  IF ( ierr == 0 ) THEN
    READ  ( UNIT=NLunit, NML=GNScodes, IOSTAT=ierr )
    READ  ( UNIT=NLunit, NML=LEOcodes, IOSTAT=ierr )
    READ  ( UNIT=NLunit, NML=ORGcodes, IOSTAT=ierr )
    CLOSE ( UNIT=NLunit )
  ELSE
    WRITE ( *, FMT="(A)" )
    WRITE ( *, FMT="(A)" ) "WARNING: ROPP-BUFR codes NAMELIST file"// &
                           " could not be opened."
    WRITE ( *, FMT="(A)" ) "         ("//TRIM(FileSpec)//")"
    WRITE ( *, FMT="(A)" ) "         Using default look-up tables."
    WRITE ( *, FMT="(A)" )
  END IF

!---------------------------------------------------
! 2. Look up numeric (BUFR) code from character (ROPP)
!---------------------------------------------------

  IF ( ind >= 1 ) THEN

! Defaults

    Lcode  = MISSING
    Icode  = MISSING
    Gclass = MISSING
    Gcode  = MISSING
    Ocode  = MISSING
    Scode  = MISSING
    Bcode  = MISSING

! LEO Rx ID code (satellite & hence instrument)

    i = nrx
    DO WHILE ( i > 0 .AND. &
               LEOlist(i) /= ROdata%LEO_id )
      i = i - 1
    END DO
    IF ( i > 0 ) THEN
      Lcode = LEOcode(i)
      Icode = Inscode(i)
    END IF

! GNSS Tx ID code (satellite class) & separate PRN

    i = ntx
    DO WHILE ( i > 0 .AND. &
               GNSlist(i) /= ROdata%GNS_id(1:1) )
      i = i - 1
    END DO
    IF ( i > 0 ) Gclass = GNScode(i)
    READ ( ROdata%GNS_id(2:4), FMT=*, IOSTAT=ierr ) Gcode
    IF ( ierr /= 0 .OR. &
         Gcode < 0 .OR. &
         Gcode > 32 ) Gcode = MISSING

! Originating (processing) centre code
! and associated sub-centre code

    i = noc
    DO WHILE ( i > 0 .AND. &
               ORGlist(i)(1:3) /= ROdata%Processing_Centre(1:3) )
      i = i - 1
    END DO
    Ocode = ORGcode(i)
    Scode = SUBcode(i)

! Look up background generator centre code

    i = nbg
    DO WHILE ( i > 0 .AND. &
               BGDlist(i)(1:3) /= ROdata%BG%Source(1:3) )
      i = i - 1
    END DO
    Bcode = BGDcode(i)

!---------------------------------------------------
! 3. Look up character (ROPP) code from numeric (BUFR)
!---------------------------------------------------

  ELSE

! Defaults

    ROdata%LEO_id            = "UNKN"
    ROdata%GNS_id            = "U999"
    ROdata%Processing_Centre = "UNKNOWN"
    ROdata%bg%Source         = "UNKNOWN"

! LEO Rx ID code (Satellite)

    i = nrx
    DO WHILE ( i > 0 .AND. &
               Lcode /= LEOcode(i) )
      i = i - 1
    END DO
    IF ( i > 0 ) ROdata%LEO_id = LEOlist(i)

! GNSS Tx ID code (from satellite class) & add PRN

    i = ntx
    DO WHILE ( i > 0 .AND. &
               Gclass /= GNScode(i) )
      i = i - 1
    END DO
    IF ( i > 0 ) ROdata%GNS_id(1:1) = GNSlist(i)
    IF ( Gcode < 0 .OR. Gcode > 999 ) Gcode = 999
    WRITE ( ROdata%GNS_id(2:4), &
            FMT="(I3.3)",       &
            IOSTAT=ierr ) Gcode

! Originating (RO processing) centre code

    i = noc
    DO WHILE ( i > 0 .AND. &
               Ocode /= ORGcode(i) )
      i = i - 1
    END DO
    IF ( i > 0 ) THEN
      j = MAX ( LEN_TRIM ( ORGlist(i) ), 4 )
      ROdata%Processing_Centre = ORGlist(i)(1:j) &
                       // " " // ORGname(i)
    END IF

! Background generating centre code

    i = nbg
    DO WHILE ( i > 0 .AND. &
               Bcode /= BGDcode(i) )
      i = i - 1
    END DO
    IF ( i > 0 ) ROdata%bg%Source = BGDlist(i)

  ENDIF

END SUBROUTINE ConvertCodes
!----------------------------------------------------------------------------
SUBROUTINE EncodeBUFR ( BUFRdsn,    & ! (in)
                        BulSeqNo,   & ! (in)
                        OrigICAO,   & ! (in)
                        OrigCentre, & ! (in)
                        SubCentre,  & ! (in)
                        descr,      & ! (inout)
                        ndescr,     & ! (inout)
                        Values,     & ! (in)
                        Names,      & ! (in)
                        nobs,       & ! (in)
                        GTShdrType, & ! (in)
                        ierr )        ! (out)
!
!****s* ropp2bufr/EncodeBUFR *
!
! NAME
!   EncodeBUFR - Encode converted RO data to BUFR message & write it out
!
! ARGUMENTS
!   BUFRdsn     (in)    chr  Output BUFR file name
!   BulSeqNo    (in)    int  Bulletin sequence number (001-999)
!   OrigICAO    (in)    chr  4-chr ICAO designator for originator centre
!   OrigCentre  (in)    int  Originator centre BUFR common code value
!   SubCentre   (in)    int  Processing centre code value
!   descr       (inout) int  On entry: array of descriptors for type
!                            On exit : array of expanded descriptors
!   ndescr      (inout) int  On entry: no. of initial descriptors in descr
!                            On exit : no. of expanded descriptors in descr
!   Values      (in)    flt  Array(ne) of converted values for BUFR encoder
!   Names       (in)    chr  List of character-based names
!   nobs        (in)    int  No. of observations
!   GTShdrType  (in)    int  GTS header type indicator
!   ierr        (out)   int  Exit code
!
! DEPENDENCIES:
!   MetDB BUFR package  - BUFR kernel routines
!   gtshdrs             - routines to add WMO/GTS routing header/trailer
!
! CALLS
!   ENBUFV2
!   GTSHDR
!   GTSEND
!   METDB_CWRITE
!
! CALLED BY
!   ropp2bufr
!
! SYNOPSIS
!    USE ropp2bufrmod
!    INTEGER :: bulseqno, origcentre, subcentre
!    INTEGER :: nobs, ndescr, gtshdrtype, ierr
!    INTEGER :: descr(nd)
!    REAL    :: values(ne,no)
!    CHARACTER (LEN=4) :: origicao
!    CHARACTER (LEN=no*4) :: names
!    CALL EncodeBUFR ( BUFRdsn, bulseqno, &
!                      origicao, origcentre, subcentre, &
!                      descr, ndescr, &
!                      values, names, nobs, gtshdrtype, ierr )
!
! DESCRIPTION
!   Encodes data in array "values" (pre-converted to BUFR
!   standard) containing "nobs" observations, to a BUFR message,
!   using the given descriptor sequence in "desc" (of length
!   "ndescr"). On exit, descr will contain an updated ndesc
!   expanded descriptor list, so this array should be sized to
!   the maximum expected expansion.
!   The data is encoded and shipped out to the file
!   specified in "BUFRdsn", which is automatically opened on
!   on the first call. The status of the encoding and file
!   output is indicated in "ierr" on return.
!   If any errors in the encoding process occur, a plain
!   text message is written to (stdout).
!   An optional GTS routing header is pre-pended (and trailer
!   bytes appended) if GTShdrType=1. Futher, 4 leading null bytes are
!   included if GTShdrType=2 for compatibility with X.25 GTS
!   transmission software, or 8-byte length + 2-byte type are
!   included if GTShdrType=3, for compatability with GTS transmission
!   via IP (ftp). No headers are generated at all if GTShdrType is not
!   1, 2 or 3.
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

  IMPLICIT NONE

! Fixed values

  INTEGER, PARAMETER :: ErrUnit = 6       ! Output stream for error messages

! Argument list parameters

  CHARACTER (LEN=*), INTENT(IN)    :: BUFRdsn       ! output BUFR file name
  INTEGER,           INTENT(IN)    :: BulSeqNo      ! Bulletin sequence number
  CHARACTER (LEN=*), INTENT(IN)    :: OrigICAO      ! originating centre ICAO code
  INTEGER,           INTENT(IN)    :: OrigCentre    ! originating centre code value
  INTEGER,           INTENT(IN)    :: SubCentre     ! processing centre code value
  INTEGER,           INTENT(INOUT) :: descr(:)      ! descriptor sequence
  INTEGER,           INTENT(INOUT) :: ndescr        ! no. of descriptors in seq.
  REAL,              INTENT(IN)    :: Values(:)     ! data to encode
  CHARACTER (LEN=*), INTENT(IN)    :: Names         ! characters to encode
  INTEGER,           INTENT(IN)    :: nobs          ! no. of observations
  INTEGER,           INTENT(IN)    :: GTShdrType    ! Code for GTS header generation
  INTEGER,           INTENT(OUT)   :: ierr          ! error status:
                                                    !  0 = OK
                                                    !  1 = BUFR encode error
                                                    !  2 = File write error

! Local variables

  CHARACTER (LEN=50000) :: Message        ! BUFR message
  INTEGER :: Edition     = 3              ! BUFR Edition (3)
  INTEGER :: MasterTable = -99            ! Master BUFR Tables (default)
  INTEGER :: OCentre                      ! Coded originating centre and sub-centre
  INTEGER :: DataType    = 3              ! Data type from Table A (Sounding - satellite)
  INTEGER :: DataSubType = 255            ! Data sub-type (dummy)
  INTEGER :: VerMasTable = 12             ! Table version number (12)
  INTEGER :: VerLocTable = -99            ! Local Table version (default)
  INTEGER :: Sect3Type   = 1              ! Observed data
  INTEGER :: DateTime(5)                  ! Date & Time of data (yr,mth,day,hr,min)
  INTEGER :: lat, lon                     ! Nominal mean ob. location
  INTEGER :: lenh, lenm                   ! Length of GTS header & BUFR message
  INTEGER :: nelem                        ! No. of elements
  INTEGER :: i                            ! Loop counter
  LOGICAL :: compress     = .FALSE.       ! Compression flag

  LOGICAL   :: ExtraSect1 = .FALSE.       ! Nothing extra
  CHARACTER :: CharSect1  = " "

  LOGICAL   :: ExtraSect2 = .FALSE.       ! Nothing extra
  CHARACTER :: CharSect2  = " "

!-------------------------------------------------------------
! 1. Initialise
!-------------------------------------------------------------

  nelem = SIZE ( Values )
  ierr  = 0

!-------------------------------------------------------------
! 2. Code processing centre as sub-centre of originating
!    centre if valid
!-------------------------------------------------------------

  OCentre = OrigCentre
  IF ( SubCentre >    0 .AND. &
       SubCentre <= 254 )     &
    OCentre = OCentre + SubCentre * 256

!-------------------------------------------------------------
! 3. Extract first (or only) observation time for BUFR header
!    and nominal lat/long for Area (A2) code in GTS header
!-------------------------------------------------------------

  DateTime(1:5) = NINT(Values(7:11))

  lat = NINT(Values(30))
  lon = NINT(Values(31))
  lon = MOD(lon+360,360) ! limit range to 0-359deg

!-------------------------------------------------------------
! 4. Generate WMO bulletin GTS routing header if required
!-------------------------------------------------------------

  Message = " "
  lenh    = 0
  CALL GTSHDR ( GTShdrType,  &
                BulSeqNo,    &
                tta,         &
                lat, lon,    &
                ii,          &
                OrigICAO,    &
                DateTime,    &
                Message,     &
                lenh )

!-------------------------------------------------------------
! 4. Do the encode
!-------------------------------------------------------------

  CALL ENBUFV2 ( descr,       Values,        &
                 ndescr,      nelem, nobs,   &
                 Names,       DateTime,      &
                 Message(lenh+1:),           &
                 compress,                   &
                 lenm,                       &
                 Edition,     MasterTable,   &
                 OCentre,                    &
                 DataType,    DataSubType,   &
                 VerMasTable, VerLocTable,   &
                 ExtraSect1,  CharSect1,     &
                 ExtraSect2,  CharSect2,     &
                 Sect3Type )

! ENBUFV2 ought to return ndescr as the no. of expanded descriptors,
! but it doesn't - the returned value is the number of data elements
! which we already know (nelem). So we scan the descr array to find
! the last valid descriptor.

  ndescr = SIZE(descr)
  DO
    IF ( ndescr       == 0 ) EXIT
    IF ( descr(ndescr) > 0 ) EXIT
    ndescr = ndescr - 1
  END DO
  IF ( DEBUG ) THEN
    WRITE ( *, FMT="(A)"    ) "Encoding results:"
    WRITE ( *, FMT="(A,I7)" ) "  No. of expanded BUFR descr.  :", ndescr
    WRITE ( *, FMT="(A,I7)" ) "  Length of BUFR message       :", lenm
  END IF

!-------------------------------------------------------------
! 5. If ok, add optional ending to message (and if IP leader,
!    insert length of message from SOH to ETX in first 8
!    bytes) & ship it out
!-------------------------------------------------------------

  IF ( lenm > 3 .AND. &
       Message(lenm+lenh-3:lenm+lenh) == "7777" ) THEN
    lenm = lenh + lenm + 1
    CALL GTSEND ( GTShdrType, &
                  Message,    &
                  lenm )

    IF ( DEBUG .AND. &
         GTSHdrType /= NOhdrs ) THEN
      WRITE ( *, "(A)", ADVANCE="NO" ) "  GTS bulletin header / ARH    : "
      DO i = 1, lenh-3
        IF ( LGE(Message(i:i)," ") ) THEN
          WRITE ( *, "(A)", ADVANCE="NO" ) Message(i:i)
        ELSE
          WRITE ( *, "(A)", ADVANCE="NO" ) "."
        END IF
      END DO
      WRITE ( *, "(A)" )
    END IF

    WRITE ( *, FMT="(A,I6,A)", ADVANCE="NO" ) "Writing", lenm, " bytes "
    IF ( GTSHdrType /= NOhdrs ) &
      WRITE ( *, FMT="(A)", ADVANCE="NO" ) "(GTS bulletin) "
    WRITE ( *, FMT="(A)" ) "to "//TRIM(BUFRdsn)//"..."
    CALL METDB_CWRITE ( BUFRunit,        &
                        Message(1:lenm), &
                        lenm )
    IF ( lenm == 0 ) ierr = 2

    IF ( ierr /= 0 ) &
      WRITE ( *, FMT="(A)" ) "ERROR: writing to BUFR file "// &
                             TRIM(BUFRdsn)
  ELSE
    WRITE ( *, FMT="(A)" ) "ERROR: generating BUFR message"
    ierr = 1
    lenm = 0
  END IF

END SUBROUTINE EncodeBUFR
!--------------------------------------------------------------------
END MODULE ropp2bufrmod
!--------------------------------------------------------------------
PROGRAM ropp2bufr

!****x* ropp2bufr/ropp2bufr *
!
! NAME
!   ropp2bufr - Encode a ROPP file to WMO FM-94 (BUFR)
!
! CALLS
!   IARGC
!   BulSeqIO
!   ConvertROPPtoBUFR
!   EncodeBUFR
!   GetOptions
!   ropp_io_ascend
!   ropp_io_occid
!   ropp_io_read
!   ropp_io_thin
!   ropp_io_free
!   IDES
!   METDB_COPEN
!   METDB_CWRITE
!   METDB_CCLOSE
!   GTSEND
!   DateTimeNow
!   DateTimeOffset
!   JulianDay
!   To_Lower
!   FileDelete
!
! MODULES
!   ropp2bufrmod        - fixed parameter definitions
!   ropp_io             - ROPP I/O file support
!   ropp_io_types       - ROPP data type definitions
!   DateTimeProgs       - Date & Time conversion routines
!   DateTimeTypes       - Date & Time conversion definitions
!
! INCLUDES
!   portability.fi      - system dependent settings (from BUFR package)
!                         to support EXIT()
!
! DEPENDENCIES
!   MetDB BUFR package  - BUFR kernel routines
!   ROPP I/O library    - ROPP file I/O support
!   ROPP Tools library  - ROPP utility routines
!   netCDF library      - netCDF file support
!   udunits library     - Units conversion routines
!
! ENVIRONMENT VARIABLES
!   BUFR_LIBRARY        - path for run-time files
!
! COMPLETION CODES
!   0 = OK
!  -1 = Occultation rejected as too old for GTS
!   1 = I/O error
!   2 = Memory allocation failure
!
! SYNOPSIS
!   > export BUFR_LIBRARY=bufr_directory_path
!   > ropp2bufr ropp_file [ropp_file...] [-o bufr_file]
!                         [-c orig_code] [-g[n]] [-s seq_file]
!                         [-p thin_file] [-t time]
!                         [-u] [-l] [m] [-h|?] [-v] [-d]
! INPUTS
!   ropp_file is the input file(s). This file must be in ROPP V1.0
!             (TEXT or netCDF) or CLIMAP V2.2 (TEXT) RO format
!             (See Refs.1,2)
! OUTPUT
!   bufr_file is the output file, which will contain one encoded
!             BUFR message (See Ref.3)
!             The output file name is optional, and if not specified,
!             is generated from the occulation ID.
!
! OPTIONS
!   Option switches can be in any order and are case-insensitive;
!   any space(s) between a switch and its (madatory) argument is
!   optional.
!     -o  specifies the BUFR output file name
!     -c  specifies the originating centre code value
!     -g  specifies that GTS routing headers/trailers are required
!     -gx speciifes that GTS headers include 4 leading null bytes
!         (required for some X.25 implimentations for GTS)
!     -gi speciifes that GTS headers include 10-byte leading size/type
!         (required for some IP (FTP) implimentations for GTS)
!     -s  specifies a bulletin sequence number file
!     -p  specifies a thinning control file or max. no. of levels
!     -t  specifies a time (age) rejection threshold
!     -u  leave profiles unordered - disables the default re-ordering
!         of output profiles to ascending.
!         NB: using -u, profiles thinned using one of the interpolation methods
!         will retain the order of the fixed levels in the control file; other
!         methods will retain the ordering of the input profiles.
!     -l  to skip encoding L1+L2 data (bending angle, Level 1b), if present
!     -m  to skip encoding met.  data (geophysical, Level 2b,c), if present
!     -d  to output additional diagnostics
!     -h or ? to output summary help
!     -v  to output program version ID
!
! DEFAULTS
!     Input file name         : ropp.nc (netDCF)
!     Output file name        : <occid>.bufr
!     Originating centre code : 74 (Bracknell)
!     GTS routing headers     : not generated
!     Bulletin sequence       : initialised at 001
!     Time threshold          : 00:00 (no cut-off) unless one of
!                               -g options present, when 23:50
!     Encode                  : all available Level 1b, 2b & 2c data
!     Thinning                : none
!
! DESCRIPTION
!   A BUFR encoder for Radio Occultation data.
!   Reads from a ROPP v1.0 (TEXT or netCDF) or CLIMAP V2.2 (TEXT)
!   formatted file and encodes data therein to one BUFR message.
!   Various options are provided to control the generation of
!   routing headers and rejection based on the age of the data
!   and to skip encoding certain profile subsets.
!   BUFR tables and other run-time files are found via the environment
!   variable 'BUFR_LIBRARY'.
!
! REFERENCES
!   1) Format Definition for Radio Occultation Files -
!      CLIMAP Format Version 2.2a
!   2) ROPP interface file format
!      SAF/GRAS/METO/FMT/ROPP/001
!   3) WMO FM94 (BUFR) Specification for GRAS SAF Processed Radio
!      Occultation Data.
!      SAF/GRAS/METO/FMT/BUFR/001
!   4) Monodimensional data thinning for GPS radio occultations
!      SAF/GRAS/METO/ALG/ROPP/001
!
! SEE ALSO
!   ropp2bufr(1), bufr2ropp(1), decbufr(1)
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

! Modules

  USE ROPP2BUFRmod
  USE ropp_io_types, ONLY: ROprof
  USE ropp_io,       ONLY: ropp_io_nrec,  &
                           ropp_io_occid, &
                           ropp_io_read,  &
                           ropp_io_thin,  &
                           ropp_io_free
  USE DateTimeProgs, ONLY: DateTimeNow,   &
                           DateTimeOffset
  USE DateTimeTypes, ONLY: DTtype,        &
                           DateFmt,       &
                           TimeFmt,       &
                           MonthName,     &
                           IdxMinute,     &
                           NHourPerDay,   &
                           NMinPerHour,   &
                           JD2000
  USE ropp_utils,    ONLY: File_Delete

! Include files

  INCLUDE "portability.fi"

! Fixed values

  CHARACTER (LEN=*), PARAMETER :: seqtype = "GPSRO" ! GPS RO sub-sequence ID
  CHARACTER (LEN=*), PARAMETER :: Fmt1 = &
                     "(A,I2.2,':',I2.2,'UT ',I2.2,'-',A3,'-',I4)"
  CHARACTER (LEN=3), PARAMETER :: Month(0:12) = &
                     (/ "???", "Jan", "Feb", "Mar", "Apr", "May", "Jun", &
                               "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" /)
! Local variables

  CHARACTER (LEN=10)  :: LastMsg        ! Last 'message' for IP

  CHARACTER (LEN=256),  &
          DIMENSION(:), &
          ALLOCATABLE :: ROPPdsn        ! Name(s) of input ROPP file(s)
  CHARACTER (LEN=256) :: BUFRdsn        ! Output (BUFR) file name
  CHARACTER (LEN=256) :: BulSeqdsn      ! bulletin sequence file name
  CHARACTER (LEN=256) :: Thindsn        ! thinning control file name
  CHARACTER (LEN=4)   :: OrigICAO       ! Originating centre ICAO code
  INTEGER             :: nfiles         ! No. of file names on command line
  INTEGER             :: OrigCentre     ! Originating centre BUFR code
  INTEGER             :: GTShdrType     ! Code for GTS header generation
  INTEGER             :: RejTimeDiff    ! reject obs older than this
  LOGICAL             :: CorrOnly       ! Flag for Corrected only
  LOGICAL             :: nomet          ! Flag for no met. data

  CHARACTER (LEN=100) :: Names = " "    ! Names list (not used)

  REAL,    ALLOCATABLE :: Values(:)     ! Data for BUFR
  INTEGER, ALLOCATABLE :: Descr(:)      ! Descriptor sequence

  INTEGER :: ndescr                     ! Number of descriptors in sequence
  INTEGER :: nelem                      ! No. of elements
  INTEGER :: nvalues                    ! No. of elements after thinning
  INTEGER :: minextra = 2000            ! Min. expansion elements for Decsr
  INTEGER :: nextra                     ! Extra expansion elements for Descr
  INTEGER :: nobs = 1                   ! No. of observations
  INTEGER :: BulSeqNo = 0               ! Bulletin Sequence No.
  INTEGER :: SubCentre                  ! Processing centre ID as code value
  INTEGER :: ierr, status               ! File error & return status codes
  INTEGER :: nmsg = 0                   ! Count of BUFR messages
  INTEGER :: LenMSG                     ! Length of BUFR messages
  INTEGER :: nfreq  = 0                 ! No. of Level 1b frequencies
  INTEGER :: nvalid = 0                 ! No. of valid L1 b/angles
  INTEGER :: JulDay                     ! Date as Julian Day no.
  INTEGER :: in, iprof, ifile           ! Loop counters
  LOGICAL :: exists                     ! File present flag
  LOGICAL :: first = .TRUE.             ! First profile flag
  LOGICAL :: unordered                  ! Enable ordering of profiles to ascending

  INTEGER :: IDES                       ! convert descriptor function

  TYPE(ROprof)       :: ROdata          ! ROPP data structure
  INTEGER            :: nprofs          ! No. of profiles in i/p file
  INTEGER            :: tprofs = 0      ! Total profiles in all files

  TYPE (DTtype)      :: DateTime        ! Date & Time structure
  CHARACTER (LEN=11) :: DateStr         ! Date string for run time
  CHARACTER (LEN=15) :: TimeStr         ! Time string for run time
  CHARACTER (LEN=10) :: NumStr          ! Number string
  INTEGER            :: MinRej          ! Rejection threshold (minutes since 00:00UT 1-Jan-2000)
  INTEGER            :: MinObs          ! Ob time (minutes since 00:00UT 1-Jan-2000)

!--------------------------------------------------------------
! 1. Begin
!--------------------------------------------------------------

  CALL DateTimeNow ( "UT", DateTime )
  WRITE ( TimeStr, FMT=TimeFmt ) DateTime%Hour,           &
                                 DateTime%Minute
  WRITE ( DateStr, FMT=DateFmt ) DateTime%Day,            &
                                 DateTime%MonthName(1:3), &
                                 DateTime%Year
  WRITE ( *, FMT="(/A/A/)" ) REPEAT(" ",13)// &
             "===== ROPP BUFR Encoder =====", &
             REPEAT(" ",19)//TimeStr(1:5)//" "//ADJUSTL(DateStr)

!--------------------------------------------------------------
! 2. Get file names & options from command line
!--------------------------------------------------------------

  nfiles = MAX ( IARGC(), 1 )
  ALLOCATE ( ROPPdsn(nfiles) )
  CALL GetOptions ( ROPPdsn,     &
                    nfiles,      &
                    BUFRdsn,     &
                    BulSeqdsn,   &
                    Thindsn,     &
                    OrigICAO,    &
                    OrigCentre,  &
                    GTShdrType,  &
                    RejTimeDiff, &
                    CorrOnly,    &
                    nomet,       &
                    unordered )

!--------------------------------------------------------------
! 3. If time rejection on, set time rejection threshold in
!    minutes since 00:00UT 1-Jan-2000 for specified period
!    back from 'now' (NB Julian Day element of DateTime
!    increments at midday, not, midnight, so we need to
!    compensate)
!--------------------------------------------------------------

  IF ( RejTimeDiff > 0 ) THEN
    CALL DateTimeOffset ( DateTime, -RejTimeDiff, IdxMinute )
    IF ( DateTime%Hour >= 12 ) DateTime%JulDay = DateTime%JulDay - 1
    MinRej = ( ( DateTime%JulDay - JD2000 ) * nHourPerDay &
           +     DateTime%Hour ) * nMinPerHour            &
           + DateTime%Minute
    IF ( DEBUG ) THEN
      WRITE ( TimeStr, FMT=TimeFmt ) DateTime%Hour,           &
                                     DateTime%Minute
      WRITE ( DateStr, FMT=DateFmt ) DateTime%Day,            &
                                     DateTime%MonthName(1:3), &
                                     DateTime%Year
      WRITE ( *, FMT="(A)" ) "Rejecting occultations older than "// &
                             TimeStr(1:5)//"UT "//ADJUSTL(DateStr)
    END IF
  ELSE
    MinRej = 0
  END IF

!--------------------------------------------------------------
! 4. If GTS headers to be generated, read the last used
!    bulletin sequence number
!--------------------------------------------------------------

  IF ( GTShdrType /= NOhdrs )  &
    CALL BulSeqIO ( BulSeqdsn, &
                    BulseqNo,  &
                    Input )

!--------------------------------------------------------------
! 5. Loop over input files
!--------------------------------------------------------------

  DO ifile = 1, nfiles

    INQUIRE ( FILE=ROPPdsn(ifile), EXIST=exists )
    IF ( .NOT. exists ) THEN
      WRITE ( *, FMT="(A/)" ) "*** ROPP input file  " // &
                              TRIM(ROPPdsn(ifile))    // &
                              " not found"
      CYCLE
    ENDIF

    WRITE ( *, FMT="(A)" ) "Reading  ROPP data from " // &
                           TRIM(ROPPdsn(ifile))

    nprofs = ropp_io_nrec ( ROPPdsn(ifile) )
    IF ( nprofs < 0 ) nprofs = 1            ! assume 1 profile for text type
    tprofs = tprofs + nprofs

!--------------------------------------------------------------
! 6. Loop over occultations from current file
!    (If a read error, skip to next file)
!--------------------------------------------------------------

    DO iprof = 1, nprofs

      CALL ropp_io_read ( ROdata,          &
                          file=ROPPdsn(ifile), &
                          rec=iprof,       &
                          ierr=status )

      IF ( status /= 0 ) THEN
        WRITE ( *, FMT="(A)" ) "*** Failed to read file"
        EXIT
      END IF

!--------------------------------------------------------------
! 7. On first profile, open output file for BUFR (default name
!    from first occultation ID)
!--------------------------------------------------------------

      CALL ropp_io_occid ( ROdata )
      IF ( first ) THEN
        IF ( BUFRdsn == " " ) THEN
          BUFRdsn = TRIM(ROdata%Occ_id) // ".bufr"
          CALL To_Lower ( BUFRdsn )
        END IF
        CALL METDB_COPEN ( BUFRunit,      &
                           TRIM(BUFRdsn), &
                           Output,        &
                           ierr )
        IF ( ierr /= 0 ) THEN
          WRITE ( *, FMT="(A)" ) "*** Failed to open BUFR output file "// &
                                 TRIM(BUFRdsn)
          CALL EXIT(ErrIO)
        END IF
        first = .FALSE.
      END IF

      WRITE ( *, FMT="(A,I4)" ) "Encoding ROPP profile", iprof
      IF ( ROdata%DTocc%Month < 1 .OR. &
           ROdata%DTocc%Month > 12 ) ROdata%DTocc%Month = 0
      WRITE ( TimeStr, FMT=TimeFmt ) ROdata%DTocc%Hour,        &
                                     ROdata%DTocc%Minute
      WRITE ( DateStr, FMT=DateFmt ) ROdata%DTocc%Day,         &
                           MonthName(ROdata%DTocc%Month)(1:3), &
                                     ROdata%DTocc%Year
      WRITE ( *, FMT="(A)" ) "Occultation Ident"//REPEAT(" ",14)// &
                             ": "//TRIM(ROdata%Occ_ID)
      WRITE ( *, FMT="(A)" ) "Processing RO profile for"//REPEAT(" ",6)// &
                         ": "//TimeStr(1:5)//"UT "//ADJUSTL(DateStr)

!--------------------------------------------------------------
! 7.1 If GTS time rejection on, skip if occultation time
!     is too old
!--------------------------------------------------------------

      IF ( MinRej > 0 ) THEN
        CALL JulianDay ( ROdata%DTocc%Year,  &
                         ROdata%DTocc%Month, &
                         ROdata%DTocc%Day,   &
                         JulDay, 1 )
        MinObs = ( ( JulDay - JD2000 )   * nHourPerDay &
               +     ROdata%DTocc%Hour ) * nMinPerHour &
               +     ROdata%DTocc%Minute
        IF ( MinObs < MinRej ) THEN
          WRITE ( *, FMT="(A)" ) "WARNING: Occultation is too old for GTS "// &
                                 "- not encoded."
          CYCLE
        END IF
      END IF

!--------------------------------------------------------------
! 7.2 Use (at most) one Level 1a data for nominal POD
!     Only encode Level 1b L1+L2 data if:
!     a) 'Corrected only' option not taken and
!     b) there is at least one valid L1 bending angle value present
!--------------------------------------------------------------

      ROdata%Lev1a%Npoints = MIN ( 1, ROdata%Lev1a%Npoints )

      IF ( .NOT. CorrOnly ) THEN
        nvalid = 0
        DO in = 1, ROdata%Lev1b%Npoints
          IF ( ROdata%Lev1b%BAngle_L1(in) > 0.0 .AND. &
               ROdata%Lev1b%BAngle_L1(in) < 0.082 ) THEN
            nvalid = nvalid + 1
          END IF
        END DO
        IF ( nvalid < 1 ) CorrOnly = .TRUE.
      END IF

      IF ( ROdata%Lev1b%Npoints > 0 ) THEN
        IF ( CorrOnly ) THEN
          nfreq = 1
        ELSE
          nfreq = 3
        END IF
      END IF

!--------------------------------------------------------------
! 7.3 Only encode 'Met' data if:
!     a) 'No Met' option not taken and
!     b) there is at least one valid temperature value present.
!     Level 2c (Surface) data is always encoded, but show as '0'
!     if not valid.
!     Ignore any level 2d data
!--------------------------------------------------------------

      nvalid = 0
      DO in = 1, ROdata%Lev2b%Npoints
        IF ( ROdata%Lev2b%Temp(in) > 150.0 .AND. &
             ROdata%Lev2b%Temp(in) < 350.0 ) THEN
          nvalid = nvalid + 1
        END IF
      END DO
      IF ( nvalid < 1 ) nomet = .TRUE.

      IF ( nomet ) THEN
        IF ( ROdata%Lev2b%Npoints > 0 ) THEN
          ROdata%Lev2b%Npoints = 0
          ROdata%Lev2c%Npoints = 0
        END IF
      END IF

      IF ( ROdata%Lev2c%Geop_Sfc < -1000.0 .OR. &
           ROdata%Lev2c%Geop_Sfc > 10000.0 )    &
           ROdata%Lev2c%Npoints = 0

      ROdata%Lev2d%Npoints = 0

! Skip this profile if no valid bending angles, refractivity,
! met. or surface met. present

      IF ( ROdata%Lev1b%Npoints <= 0 .AND. &
           ROdata%Lev2a%Npoints <= 0 .AND. &
           ROdata%Lev2b%Npoints <= 0 .AND. &
           ROdata%Lev2c%Npoints <= 0 ) THEN
        WRITE ( *, FMT="(A)" ) "WARNING: No. of L1b,2a,2b,2c samples" // &
                               " all zero - skipping this profile"
        CYCLE
      END IF

!--------------------------------------------------------------
! 7.4 Thin BA, N & T,q,p profiles as required; ensure all
!     profiles to be encoded are in ascending height order
!--------------------------------------------------------------

      CALL ropp_io_thin ( ROdata,    &
                          Thindsn,   &
                          DEBUG )
      IF ( .NOT. unordered ) THEN
        IF ( DEBUG ) WRITE ( *, "(A)" ) "Ensuring all profiles are in ascending height order..."
        CALL ropp_io_ascend ( ROdata )
      END IF

!--------------------------------------------------------------
! 7.5 Calculate total number of BUFR elements for this profile
!     and allocate working array for BUFR-interface data values
!--------------------------------------------------------------

! No. of BUFR elements expected

      nelem = 37                                            &  ! Header
            +  1 + ROdata%Lev1b%Npoints * ( 5 + nfreq * 6 ) &  ! Level 1b
            +  1 + ROdata%Lev2a%Npoints * 6                 &  ! Level 2a
            +  1 + ROdata%Lev2b%Npoints * 10                &  ! Level 2b
            +  7                                               ! Level 2c

      ALLOCATE ( Values(1:nelem), STAT=status )
      IF ( status /= 0 ) THEN
        WRITE ( *, FMT="(A)" ) "ERROR: Failed to allocate memory for Values array"
        CALL EXIT(ErrMem)
      END IF
      Values(:) = 0.0

!--------------------------------------------------------------
! 7.6 Convert RO data to BUFR array
!--------------------------------------------------------------

      CALL ConvertROPPtoBUFR ( ROdata, &
                               CorrOnly,      &
                               SubCentre,     &
                               Values,        &
                               nvalues )

!--------------------------------------------------------------
! 7.7 Allocate working array for descriptors, based on thinned
!     number of data values, and allowing headroom for expansion.
!--------------------------------------------------------------

      nextra = MAX(minextra,(nvalues*5/10))
      ALLOCATE ( Descr(1:nvalues+nextra), STAT=status )
      IF ( status /= 0 ) THEN
        WRITE ( *, FMT="(A)" ) "ERROR: Failed to allocate memory for Descr array"
        CALL EXIT(ErrMem)
      END IF
      Descr(:) = 0

      IF ( DEBUG ) THEN
        WRITE ( *, FMT="(A)"    ) "Encoding the following data:"
        IF ( ROdata%GeoRef%Lon > 180.0 ) &
             ROdata%GeoRef%Lon = ROdata%GeoRef%Lon - 360.0
        WRITE ( *, FMT="(A,2F9.2)" ) "  Nominal occ lat/lon location :", &
                                  ROdata%GeoRef%Lat, ROdata%GeoRef%Lon
        WRITE ( *, FMT="(A,I7)" ) "  No. of orbit state vectors   :", &
                                  ROdata%Lev1a%Npoints
        WRITE ( *, FMT="(A,I7)" ) "  No. of bending angle samples :", &
                                  ROdata%Lev1b%Npoints
        IF ( ROdata%Lev1b%Npoints > 0 ) THEN
          WRITE ( *, FMT="(A)", ADVANCE="NO" ) &
                              "  Bending angles to encode     : "
          IF ( CorrOnly ) THEN
            WRITE ( *, FMT="(A)" ) "Corrected only"
          ELSE
            WRITE ( *, FMT="(A)" ) "L1+L2+Corrected"
          END IF
        END IF
        WRITE ( *, FMT="(A,I7)" ) "  No. of refractivity  samples :", &
                                  ROdata%Lev2a%Npoints
        WRITE ( *, FMT="(A,I7)" ) "  No. of geophysical   samples :", &
                                  ROdata%Lev2b%Npoints
        WRITE ( *, FMT="(A,I7)" ) "  No. of surface geo.  samples :", &
                                  ROdata%Lev2c%Npoints
        WRITE ( *, FMT="(A,I7)" ) "  No. of model coeff.  levels  :", &
                                  ROdata%Lev2d%Npoints

        IF ( nvalues == nelem ) THEN
          WRITE ( *, FMT="(A,I7)" ) "  Total no. of BUFR elements   :", &
                                    nelem
        ELSE
          WRITE ( *, FMT="(A,I7)" ) "  Thinned no. of BUFR elements :", &
                                    nvalues
        END IF
        WRITE ( *, FMT="(A,I7)") "  Allocated Descriptor space   :", &
                                 nvalues+nextra
      END IF

!--------------------------------------------------------------
! 7.8 Encode this occultation & write it to output BUFR file
!--------------------------------------------------------------

      Descr(1) = IDES(ROdescr)
      ndescr   = 1
      BulSeqNo = MOD ( BulSeqNo, 999 ) + 1     ! increment in range 001-999
      CALL EncodeBUFR ( BUFRdsn,    &
                        BulSeqNo,   &
                        OrigICAO,   &
                        OrigCentre, &
                        SubCentre,  &
                        Descr,      &
                        ndescr,     &
                        Values,     &
                        Names,      &
                        nobs,       &
                        GTShdrType, &
                        ierr )
      IF ( ierr == 0 ) nmsg = nmsg + 1

      IF ( ALLOCATED ( Values ) ) DEALLOCATE ( Values )
      IF ( ALLOCATED ( Descr  ) ) DEALLOCATE ( Descr )

      WRITE ( *, * )
    END DO                       ! end of profiles loop

!--------------------------------------------------------------
! 7.9 Free memory ready for next file
!--------------------------------------------------------------

    CALL ropp_io_free ( ROdata )

  END DO                         ! end of file loop

!--------------------------------------------------------------
! 8. Generate & output end-of-file dummy bulletin for IP
!    if required.
!    Close output file. Delete it if no messages were written
!--------------------------------------------------------------

  IF ( GTShdrType == IPhdrs .AND. &
       nmsg       >  0 ) THEN
    LenMsg = 0
    CALL GTSEND ( GTShdrType, &
                  LastMsg,    &
                  LenMsg )

    WRITE ( *, FMT="(A,I6,A/)" ) "Writing", LenMsg,       &
                                 " bytes (for EOF) to "// &
                                 TRIM(BUFRdsn)//"..."
    CALL METDB_CWRITE ( BUFRunit,          &
                        LastMsg(1:LenMsg), &
                        LenMsg )
  ENDIF

  CALL METDB_CCLOSE ( BUFRunit )

  IF ( nmsg == 0 ) THEN
    CALL file_delete ( BUFRdsn, ierr )
  END IF


!--------------------------------------------------------------
! 9. If GTS headers were generated, save last used bulletin
!    sequence number
!--------------------------------------------------------------

  IF ( GTShdrType /= NOhdrs )  &
    CALL BulSeqIO ( BulSeqdsn, &
                    BulseqNo,  &
                    Output )

!--------------------------------------------------------------
! 10. Tidy up & finish
!--------------------------------------------------------------

  IF ( nmsg == 0 ) THEN
    WRITE ( *, FMT="(A)" ) "WARNING: No profiles were encoded "// &
                            "or written to the BUFR file"
  ELSE IF ( nmsg < tprofs ) THEN
    WRITE ( *, FMT="(A)" ) "WARNING: Some profiles were not encoded "// &
                            "or written to the BUFR file"
  END IF

  IF ( nmsg > 0 ) THEN
    WRITE ( Numstr, FMT="(I10)" ) nmsg
    WRITE ( *, FMT="(A)", ADVANCE="NO" ) "Generated " // &
                                          TRIM(ADJUSTL(Numstr))
    IF ( GTShdrType == NOhdrs ) THEN
      WRITE ( *, FMT="(A)", ADVANCE="NO" ) " BUFR messages"
    ELSE
      WRITE ( *, FMT="(A)", ADVANCE="NO" ) " GTS bulletins"
    END IF
    WRITE ( *, FMT="(A)" ) " to " // TRIM(BUFRdsn)
  END IF

  IF ( ALLOCATED(ROPPdsn) ) DEALLOCATE (ROPPdsn)
  WRITE ( *, * )
  CALL EXIT(ErrOK)

END PROGRAM ropp2bufr
