! $Id: ropp_io_types.f90 1834 2008-10-13 16:05:55Z frhl $

module ropp_io_types

!****m* Modules/ropp_io_types *
!
! NAME
!    ropp_io_types - Type declarations for the ROPP IO library.
!
! SYNOPSIS
!    use ropp_io
!
! DESCRIPTION
!    This Fortran module supports the ROPP input / output library and provides
!    derived data types used by the ropp_io library. It must be loaded for all
!    applications using the ropp_io library / package. Note that loading the
!    ropp_io module includes the ropp_io_types module
!
! SEE ALSO
!    ropp_io_types
!    ropp_io_init
!    ropp_io_free
!    ropp_io_read
!    ropp_io_write
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

!-------------------------------------------------------------------------------
! 1. Other modules
!-------------------------------------------------------------------------------

  use typesizes, wp => EightByteReal

!-------------------------------------------------------------------------------
! 2. Useful variables
!-------------------------------------------------------------------------------

!****ip* Initialisation/ThisFmtVer
!
! NAME
!    ThisFmtVer - Parameter describing the current implemented ROPP 
!                 data format version (NOT the current software 
!                 version of ROPP).
!
! NOTES
!    This parameter is an internal parameter to the ropp_io library and not
!    intended to be used by users. Update the value only if the (text) file
!    format specification changes (see Ref).
!
! REFERENCE
!    ROPP Interface File Format. SAF/GRAS/METO/FMT/ROPP/001
!
! SOURCE
!
  character (len=*), parameter :: ThisFmtVer = "ROPP I/O V1.1"
!
!****

!****ip* Initialisation/ropp_io_ncopensep
!
! NAME
!    ropp_io_ncopensep - Internal global variable keeping track if netCDF data
!                           files have been opened separately (as opposed to
!                           be opened from ropp_io_read() or ropp_io_write()).
!
! NOTES
!    This variable is an internal variable to the ropp_io library and not
!    intended to be used by users.
!
! SOURCE
!
  logical                      :: ropp_io_ncopensep = .false.
!
!****

!-------------------------------------------------------------------------------
! 3. Missing Data Flag Value
!-------------------------------------------------------------------------------

!****ip* Initialisation/ropp_io_mdfv
!
! NAME
!    ropp_io_mdfv - Internal global 'Missing Data Flag/Test Value(s)'
!
! NOTES
!    These parametres are used to indicate/test a 'missing' (invalid)
!    data value.
!    ropp_io_mdfv should be used to set invalid data for most
!      ROPP parametres, but a single universal value is not suitable for
!      all; some - e.g. (X,Y,Z) coordinate vectors - are set to zero.
!      For others, parameter-specific values may be more appropriate.
!    ropp_io_zero can be used to set parametres to zero.
!    ropp_io_mdtv can used for testing for invalid parameter values;
!      anything less than this value can be assumed to be set 'missing',
!      though again, some parametres may have specific values to test for.
!    ropp_zdtv can be used to test for (almost) zero, e.g.
!      if ( abs(value) < ropp_io_zdtv ) then ...
!         ! value can be considered to be zero
!    The values for ropp_io_mdfv and ropp_mdtv here may be changed
!    in the future when the text files are finally dead and formatting
!    limitations ceases to be an issue.
!
! SOURCE
!
  real(wp), parameter              :: ropp_io_mdfv = -9999.9_wp
  real(wp), parameter              :: ropp_io_zero =     0.0_wp

  real(wp), parameter              :: ropp_io_mdtv = -9999.0_wp
  real(wp), parameter              :: ropp_io_zdtv =   1e-10_wp
!
!****

!-------------------------------------------------------------------------------
! 4. PCD flags
!-------------------------------------------------------------------------------

!****p* Initialisation/PCD_parametres
!
! NAME
!    PCD parametres - Parametres for setting and testing the Product Confidence
!                     Data (PCD) element of the ROprof structure.
!
! SOURCE
!
  integer, parameter :: PCD_summary      =  0  ! Nominal / non-nominal quality
  integer, parameter :: PCD_offline      =  1  ! NRT / offline product
  integer, parameter :: PCD_rising       =  2  ! Setting / rising occultation
  integer, parameter :: PCD_phase        =  3  ! Phase processing      nominal / non-nominal
  integer, parameter :: PCD_bangle       =  4  ! Bending angle proc.   nominal / non-nominal
  integer, parameter :: PCD_refrac       =  5  ! Refractivity proc.    nominal / non-nominal
  integer, parameter :: PCD_met          =  6  ! Meteorological. proc. nominal / non-nominal
  integer, parameter :: PCD_open_loop    =  7  ! Open Loop data used   no / yes
  integer, parameter :: PCD_reflections  =  8  ! Surface reflections detected no / yes
  integer, parameter :: PCD_l2_signal    =  9  ! L2 GPS signal used    L2P / L2C
  integer, parameter :: PCD_reserved_11  = 10  ! Reserved
  integer, parameter :: PCD_reserved_12  = 11  ! Reserved
  integer, parameter :: PCD_reserved_13  = 12  ! Reserved
  integer, parameter :: PCD_bg           = 13  ! Background profile nominal / non-nominal
  integer, parameter :: PCD_occultation  = 14  ! Occultation / background profile
  integer, parameter :: PCD_missing      = 15  ! PCD missing
!
!****

!-------------------------------------------------------------------------------
! 4. Date and time
!-------------------------------------------------------------------------------

!****d* Datatypes/DT7units
!
! NAME
!    DT7units - A sub-structure of the DT7type structure, defining units.
!
! SOURCE
!
  type DT7units
    character(len = 64) :: year     = "years"
    character(len = 64) :: month    = "months"
    character(len = 64) :: day      = "days"
    character(len = 64) :: hour     = "hours"
    character(len = 64) :: minute   = "minutes"
    character(len = 64) :: second   = "seconds"
    character(len = 64) :: msec     = "milliseconds"
  end type DT7units
!
!****

!****d* Datatypes/DT7range
!
! NAME
!    DT7range - A sub-structure of the DT7type structure, setting valid ranges.
!
! SOURCE
!
  type DT7range
    integer, dimension(2) :: year   = (/1995, 2099/)
    integer, dimension(2) :: month  = (/  01,   12/)
    integer, dimension(2) :: day    = (/  01,   31/)
    integer, dimension(2) :: hour   = (/  00,   23/)
    integer, dimension(2) :: minute = (/  00,   59/)
    integer, dimension(2) :: second = (/  00,   59/)
    integer, dimension(2) :: msec   = (/  00,  999/)
  end type DT7range
!
!****

!****d* Datatypes/DT7type
!
! NAME
!    DT7types - A sub-structure of the ROprofs structure for date information.
!
! SOURCE
!
  type DT7type
    integer        :: year   = 9999
    integer        :: month  =   99
    integer        :: day    =   99
    integer        :: hour   =   99
    integer        :: minute =   99
    integer        :: second =   99
    integer        :: msec   =  999
    type(DT7units) :: units
    type(DT7range) :: range
  end type DT7type
!
!****

!-------------------------------------------------------------------------------
! 5. Georeferencing of the profile
!-------------------------------------------------------------------------------

!****d* Datatypes/GEOunits
!
! NAME
!    GEOunits - A sub-structure of the GEOtype structure, setting units.
!
! SOURCE
!
  type GEOunits
    character(len = 64) :: time_offset = "seconds"
    character(len = 64) :: lat         = "degrees_north"
    character(len = 64) :: lon         = "degrees_east"
    character(len = 64) :: roc         = "metres"
    character(len = 64) :: r_coc       = "metres"
    character(len = 64) :: azimuth     = "degrees_T"
    character(len = 64) :: undulation  = "metres"
  end type GEOunits
!
!****

!****d* Datatypes/GEOrange
!
! NAME
!    GEOrange - A sub-structure of the GEOtype structure, setting valid ranges.
!
! SOURCE
!
  type GEOrange
    real(wp), dimension(2) :: time_offset = (/0.0_wp,    239.999_wp/)
    real(wp), dimension(2) :: lat         = (/ -90.0_wp,  90.0_wp/)
    real(wp), dimension(2) :: lon         = (/-180.0_wp, 180.0_wp/)
    real(wp), dimension(2) :: roc         = (/ 6.2e6_wp, 6.6e6_wp/)
    real(wp), dimension(2) :: r_coc       = (/-5.0e4_wp, 5.0e4_wp/)
    real(wp), dimension(2) :: azimuth     = (/   0.0_wp, 360.0_wp/)
    real(wp), dimension(2) :: undulation  = (/-150.0_wp, 150.0_wp/)
  end type GEOrange
!
!****

!****d* Datatypes/GEOref
!
! NAME
!    GEOref - A sub-structure of the GEOtype structure, defining the reference frame
!             for the centre of curvature.
!
! SOURCE
!
  type GEOref
    character(len = 3)     :: r_coc = 'ECF'
  end type GEOref
!
!****

!****d* Datatypes/GEOtype
!
! NAME
!    GEOtype - A sub-structure of the ROprof structure, describing the georeferencing
!              a given profile.
!
! SOURCE
!
  type GEOtype
    real(wp)               :: time_offset = ropp_io_zero ! Time since start (s)
    real(wp)               :: lat         = ropp_io_mdfv ! Latitude  (deg)
    real(wp)               :: lon         = ropp_io_mdfv ! Longitude (deg)
    real(wp)               :: roc         = ropp_io_mdfv ! RoC value (m)
    real(wp), dimension(3) :: r_coc       = ropp_io_zero ! RoC offset X,Y,Z vector (m)
    real(wp)               :: azimuth     = ropp_io_mdfv ! GNSS->LEO line of sight angle (degT)
    real(wp)               :: undulation  = ropp_io_mdfv ! Geoid undulation (EGM96-WGS84) (m)
    type(GEOunits)         :: units
    type(GEOrange)         :: range
    type(GEOref)           :: reference_frame
  end type GEOtype
!
!****

!-------------------------------------------------------------------------------
! 6. Background meta data
!-------------------------------------------------------------------------------

!****d* Datatypes/BGunits
!
! NAME
!    BGunits - A sub-structure of the BGtype structure, defining units.
!
! SOURCE
!
  type BGunits
    character(len = 64) :: year     = "years"
    character(len = 64) :: month    = "months"
    character(len = 64) :: day      = "days"
    character(len = 64) :: hour     = "hours"
    character(len = 64) :: minute   = "minutes"
    character(len = 64) :: fcPeriod = "hours"
  end type BGunits
!
!****

!****d* Datatypes/BGrange
!
! NAME
!    BGrange - A sub-structure of the BGtype structure, setting valid ranges.
!
! SOURCE
!
  type BGrange
    integer,  dimension(2) :: year     = (/1995,    2099/)
    integer,  dimension(2) :: month    = (/  01,      12/)
    integer,  dimension(2) :: day      = (/  01,      31/)
    integer,  dimension(2) :: hour     = (/  00,      23/)
    integer,  dimension(2) :: minute   = (/  00,      59/)
    real(wp), dimension(2) :: fcperiod = (/  00.0_wp, 24.0_wp/)
  end type BGrange
!
!****

!****d* Datatypes/BGtype
!
! NAME
!    BGunits - A sub-structure of the ROprof structure, describing background (as used
!              in the retrieval) meta data.
!
! SOURCE
!
  type BGtype
    character(len = 20) :: source   = 'NONE' ! Source of b/g profile
    integer             :: year     = 9999   ! VT year   of b/g
    integer             :: month    =   99   ! VT month  of b/g
    integer             :: day      =   99   ! VT day    of b/g
    integer             :: hour     =   99   ! VT hour   of b/g
    integer             :: minute   =   99   ! VT minute of b/g
    real(wp)            :: fcperiod =  999.9 ! F/c period (hrs)
    type(BGunits)       :: units
    type(BGrange)       :: range
  end type BGtype
!
!****

!-------------------------------------------------------------------------------
! 7. Level 1a - Orbits, phases and amplitudes
!-------------------------------------------------------------------------------

!****d* Datatypes/L1aunits
!
! NAME
!    L1aunits - A sub-structure of the L1atype structure, defining units.
!
! SOURCE
!
  type L1aunits
    character(len = 64) :: dtime      = "seconds"
    character(len = 64) :: snr        = "volt / volt"
    character(len = 64) :: phase      = "metres"
    character(len = 64) :: r_gns      = "metres"
    character(len = 64) :: v_gns      = "metres / seconds"
    character(len = 64) :: r_leo      = "metres"
    character(len = 64) :: v_leo      = "metres / seconds"
    character(len = 64) :: phase_qual = "percent"
  end type L1aunits
!
!****

!****d* Datatypes/L1arange
!
! NAME
!    L1arange - A sub-structure of the L1atype structure, setting valid ranges.
!
! SOURCE
!
  type L1arange
    real(wp), dimension(2) :: dtime      = (/  0.0_wp,  239.999_wp/)
    real(wp), dimension(2) :: snr        = (/  0.0_wp, 2000.0_wp/)
    real(wp), dimension(2) :: phase      = (/-9999.0_wp, 10000.0_wp/)
    real(wp), dimension(2) :: r_gns      = (/-43.0e6_wp, 43.0e6_wp/)
    real(wp), dimension(2) :: v_gns      = (/ -1.0e4_wp,  1.0e4_wp/)
    real(wp), dimension(2) :: r_leo      = (/-10.0e6_wp, 10.0e6_wp/)
    real(wp), dimension(2) :: v_leo      = (/ -1.0e4_wp,  1.0e4_wp/)
    real(wp), dimension(2) :: phase_qual = (/  0.0_wp,  100.0_wp/)
  end type L1arange
!
!****

!****d* Datatypes/L1aref
!
! NAME
!    L1aref - A sub-structure of the L1atype structure, defining the reference frame
!             for POD data.
!
! SOURCE
!
  type L1aref
    character(len = 3) :: r_gns = "ECF"
    character(len = 3) :: v_gns = "ECI"
    character(len = 3) :: r_leo = "ECF"
    character(len = 3) :: v_leo = "ECI"
  end type L1aref
!
!****

!****d* Datatypes/L1atype
!
! NAME
!    L1atype - A sub-structure of the ROprof structure, containing amplitude, phase and
!              POD data.
!
! SOURCE
!
  type L1atype
    integer                           :: Npoints    = 0       ! No. of samples in L1a profile
    real(wp), dimension(:),   pointer :: dtime      => null() ! Time since start (s)
    real(wp), dimension(:),   pointer :: snr_L1ca   => null() ! Signal-to-noise ratio - L1 (V/V)
    real(wp), dimension(:),   pointer :: snr_L1p    => null() ! Signal-to-noise ratio - L1 (V/V)
    real(wp), dimension(:),   pointer :: snr_L2p    => null() ! Signal-to-noise ratio - L1 (V/V)
    real(wp), dimension(:),   pointer :: phase_L1   => null() ! Excess phase   - L1 (m)
    real(wp), dimension(:),   pointer :: phase_L2   => null() ! Excess phase   - L2 (m)
    real(wp), dimension(:,:), pointer :: r_gns      => null() ! GNSS position (m)
    real(wp), dimension(:,:), pointer :: v_gns      => null() ! GNSS velocity (m/s)
    real(wp), dimension(:,:), pointer :: r_leo      => null() ! LEO  position (m)
    real(wp), dimension(:,:), pointer :: v_leo      => null() ! LEO velocity  (m/s)
    real(wp), dimension(:),   pointer :: phase_qual => null() ! Quality value (%)
    type(L1aunits)                    :: units
    type(L1arange)                    :: range
    type(L1aref)                      :: reference_frame
  end type L1atype
!
!****

!-------------------------------------------------------------------------------
! 8. Level 1b - Bending angles and impact parametres
!-------------------------------------------------------------------------------

!****d* Datatypes/L1bunits
!
! NAME
!    L1bunits - A sub-structure of the L1btype structure, defining units.
!
! SOURCE
!
  type L1bunits
    character(len = 64)             :: lat_tp       = "degrees_north"
    character(len = 64)             :: lon_tp       = "degrees_east"
    character(len = 64)             :: azimuth_tp   = "degrees"
    character(len = 64)             :: impact       = "metres"
    character(len = 64)             :: bangle       = "radians"
    character(len = 64)             :: bangle_sigma = "radians"
    character(len = 64)             :: bangle_qual  = "percent"
  end type L1bunits
!
!****

!****d* Datatypes/L1brange
!
! NAME
!    L1brange - A sub-structure of the L1btype structure, setting valid ranges.
!
! SOURCE
!
  type L1brange
    real(wp), dimension(2)          :: lat_tp          = (/ -90.0_wp,   90.0_wp/)
    real(wp), dimension(2)          :: lon_tp          = (/-180.0_wp,  180.0_wp/)
    real(wp), dimension(2)          :: azimuth_tp      = (/   0.0_wp,  360.0_wp/)
    real(wp), dimension(2)          :: impact          = (/   6.2e6_wp,  6.6e6_wp/)
    real(wp), dimension(2)          :: bangle          = (/  -0.001_wp,   0.1_wp/)
    real(wp), dimension(2)          :: bangle_sigma    = (/   0.0_wp,    0.01_wp/)
    real(wp), dimension(2)          :: bangle_qual     = (/   0.0_wp,  100.0_wp/)
  end type L1brange
!
!****

!****d* Datatypes/L1bType
!
! NAME
!    L1btype - A sub-structure of the ROprof structure, containing bending angle and
!              impact parameter data.
!
! SOURCE
!
  type L1btype
    integer                         :: Npoints          = 0       ! No. of samples in L1b profile
    real(wp), dimension(:), pointer :: lat_tp           => null() ! Latitude  (deg)
    real(wp), dimension(:), pointer :: lon_tp           => null() ! Longitude (deg)
    real(wp), dimension(:), pointer :: azimuth_tp       => null() ! GNSS->LEO line of sight angle (degT)
    real(wp), dimension(:), pointer :: impact_L1        => null() ! Impact param   - L1   (m)
    real(wp), dimension(:), pointer :: impact_L2        => null() ! Impact param   - L2   (m)
    real(wp), dimension(:), pointer :: impact           => null() ! Impact param   - corr (m)
    real(wp), dimension(:), pointer :: impact_Opt       => null() ! Impact param   - opt  (m)
    real(wp), dimension(:), pointer :: bangle_L1        => null() ! Bending angle  - L1   (rad)
    real(wp), dimension(:), pointer :: bangle_L2        => null() ! Bending angle  - L2   (rad)
    real(wp), dimension(:), pointer :: bangle           => null() ! Bending angle  - corr (rad)
    real(wp), dimension(:), pointer :: bangle_Opt       => null() ! Bending angle  - opt  (rad)
    real(wp), dimension(:), pointer :: bangle_L1_sigma  => null() ! Error in BA    - L1   (rad)
    real(wp), dimension(:), pointer :: bangle_L2_sigma  => null() ! Error in BA    - L2   (rad)
    real(wp), dimension(:), pointer :: bangle_sigma     => null() ! Error in BA    - corr (rad)
    real(wp), dimension(:), pointer :: bangle_Opt_sigma => null() ! Error in BA    - opt   (rad)
    real(wp), dimension(:), pointer :: bangle_L1_qual   => null() ! Quality values - L1   (%)
    real(wp), dimension(:), pointer :: bangle_L2_qual   => null() ! Quality values - L2   (%)
    real(wp), dimension(:), pointer :: bangle_qual      => null() ! Quality values - corr (%)
    real(wp), dimension(:), pointer :: bangle_Opt_qual  => null() ! Quality values - opt  (%)
    type(L1bunits)                  :: units
    type(L1brange)                  :: range
  end type L1btype
!
!****

!-------------------------------------------------------------------------------
! 9. Level 2a - Refractivity
!-------------------------------------------------------------------------------

!****d* Datatypes/L2aunits
!
! NAME
!    L2aunits - A sub-structure of the L2atype structure, defining units.
!
! SOURCE
!
  type L2aunits
    character(len = 64)             :: alt_refrac   = "metres"
    character(len = 64)             :: geop_refrac  = "geopotential metres"
    character(len = 64)             :: refrac       = "N-units"
    character(len = 64)             :: refrac_sigma = "N-units"
    character(len = 64)             :: refrac_qual  = "percent"
  end type L2aunits
!
!****

!****d* Datatypes/L2arange
!
! NAME
!    L2arange - A sub-structure of the L2atype structure, setting valid ranges.
!
! SOURCE
!
  type L2arange
    real(wp), dimension(2)          :: alt_refrac   = (/-1.0e3_wp, 1.0e5_wp/)
    real(wp), dimension(2)          :: geop_refrac  = (/-1.0e3_wp, 1.0e5_wp/)
    real(wp), dimension(2)          :: refrac       = (/ 0.0_wp, 500.0_wp/)
    real(wp), dimension(2)          :: refrac_sigma = (/ 0.0_wp,  10.0_wp/)
    real(wp), dimension(2)          :: refrac_qual  = (/ 0.0_wp, 100.0_wp/)
  end type L2arange
!
!****

!****d* Datatypes/L2atype
!
! NAME
!    L2atype - A sub-structure of the ROprof structure, containing refractivity, altitude
!              and geopotential height data.
!
! SOURCE
!
  type L2atype
    integer                         :: Npoints      = 0       ! No. of samples in L2a profile
    real(wp), dimension(:), pointer :: alt_refrac   => null() ! Geometric height (m)
    real(wp), dimension(:), pointer :: geop_refrac  => null() ! Geopotential height (m)
    real(wp), dimension(:), pointer :: refrac       => null() ! Refractivity (N-units)
    real(wp), dimension(:), pointer :: refrac_sigma => null() ! Est. error in refractivity (N-units)
    real(wp), dimension(:), pointer :: refrac_qual  => null() ! Quality value (%)
    type(L2aunits)                  :: units
    type(L2arange)                  :: range
  end type L2atype
!
!****

!-------------------------------------------------------------------------------
! 10. Level 2b - Meteorological quantities
!-------------------------------------------------------------------------------

!****d* Datatypes/L2bunits
!
! NAME
!    L2bunits - A sub-structure of the L2btype structure, defining units.
!
! SOURCE
!
  type L2bunits
    character(len = 64)             :: geop         = "geopotential metres"
    character(len = 64)             :: geop_sigma   = "geopotential metres"
    character(len = 64)             :: press        = "hPa"
    character(len = 64)             :: press_sigma  = "hPa"
    character(len = 64)             :: temp         = "kelvin"
    character(len = 64)             :: temp_sigma   = "kelvin"
    character(len = 64)             :: shum         = "gram / kilogram"
    character(len = 64)             :: shum_sigma   = "gram / kilogram"
    character(len = 64)             :: meteo_qual   = "percent"
  end type L2bunits
!
!****

!****d* Datatypes/L2brange
!
! NAME
!    L2brange - A sub-structure of the L2btype structure, setting valid ranges.
!
! SOURCE
!
  type L2brange
    real(wp), dimension(2)          :: geop         = (/ -1.0e3_wp,   1.0e5_wp/)
    real(wp), dimension(2)          :: geop_sigma   = (/  0.0_wp,   500.0_wp/)
    real(wp), dimension(2)          :: press        = (/  0.01_wp, 1100.0_wp/)
    real(wp), dimension(2)          :: press_sigma  = (/  0.0_wp,     5.0_wp/)
    real(wp), dimension(2)          :: temp         = (/150.0_wp,   350.0_wp/)
    real(wp), dimension(2)          :: temp_sigma   = (/  0.0_wp,     5.0_wp/)
    real(wp), dimension(2)          :: shum         = (/  0.0_wp,    50.0_wp/)
    real(wp), dimension(2)          :: shum_sigma   = (/  0.0_wp,     5.0_wp/)
    real(wp), dimension(2)          :: meteo_qual   = (/  0.0_wp,   100.0_wp/)
  end type L2brange
!
!****

!****d* Datatypes/L2btype
!
! NAME
!    L2btype - A sub-structure of the ROprof structure, containing meteorological (i.e.
!              temperature, pressure, (specific) humidity and geopotential height data.
!
! SOURCE
!
  type L2btype
    integer                         :: Npoints     = 0       ! No. of samples in L2b profile
    real(wp), dimension(:), pointer :: geop        => null() ! Geopotential height (m)
    real(wp), dimension(:), pointer :: geop_sigma  => null() ! Est. Error in geopotential height (m)
    real(wp), dimension(:), pointer :: press       => null() ! Pressure (hPa)
    real(wp), dimension(:), pointer :: press_sigma => null() ! Est. Error in pressure (hPa)
    real(wp), dimension(:), pointer :: temp        => null() ! Temperature (K)
    real(wp), dimension(:), pointer :: temp_sigma  => null() ! Est. error in temperature (K)
    real(wp), dimension(:), pointer :: shum        => null() ! Specific humidity (g/Kg)
    real(wp), dimension(:), pointer :: shum_sigma  => null() ! Est. error in SH  (g/Kg)
    real(wp), dimension(:), pointer :: meteo_qual  => null() ! Quality value (%)
    type(L2bunits)                  :: units
    type(L2brange)                  :: range
  end type L2btype
!
!****

!-------------------------------------------------------------------------------
! 11. Level 2c - Meteorological surface quantities
!-------------------------------------------------------------------------------

!****d* Datatypes/L2cunits
!
! NAME
!    L2cunits - A sub-structure of the L2ctype structure, defining units.
!
! SOURCE
!
  type L2cunits
    character(len = 64) :: geop_sfc        = "geopotential metres"
    character(len = 64) :: press_sfc       = "hPa"
    character(len = 64) :: press_sfc_sigma = "hPa"
    character(len = 64) :: press_sfc_qual  = "percent"
  end type L2cunits
!
!****

!****d* Datatypes/L2crange
!
! NAME
!    L2crange - A sub-structure of the L2ctype structure, setting valid ranges.
!
! SOURCE
!
  type L2crange
    real(wp), dimension(2) :: geop_sfc        = (/ -1.0e3_wp,  1.0e4_wp/)
    real(wp), dimension(2) :: press_sfc       = (/250.0_wp, 1100.0_wp/)
    real(wp), dimension(2) :: press_sfc_sigma = (/  0.0_wp,    5.0_wp/)
    real(wp), dimension(2) :: press_sfc_qual  = (/  0.0_wp,  100.0_wp/)
  end type L2crange
!
!****

!****d* Datatypes/L2ctype
!
! NAME
!    L2ctype - A sub-structure of the ROprof structure, containing meteorological
!              surface pressure.
!
! SOURCE
!
  type L2ctype
    integer        :: Npoints         = 0            ! No. of samples in profile (0 or 1)
    real(wp)       :: geop_sfc        = ropp_io_mdfv ! Geopotential height of surface (m)
    real(wp)       :: press_sfc       = ropp_io_mdfv ! Surface pressure (hPa)
    real(wp)       :: press_sfc_sigma = ropp_io_mdfv ! Est. error in surface pressure (hPa)
    real(wp)       :: press_sfc_qual  = ropp_io_mdfv ! Quality value for L2b+c (%)
    type(L2cunits) :: units
    type(L2crange) :: range
  end type L2ctype
!
!****

!-------------------------------------------------------------------------------
! 12. Level 2d - Meteorological model level coefficients
!-------------------------------------------------------------------------------

!****d* Datatypes/L2dunits
!
! NAME
!    L2dunits - A sub-structure of the L2dtype structure, defining units.
!
! SOURCE
!
  type L2dunits
    character(len = 40) :: level_coeff_a = 'hPa'
    character(len = 40) :: level_coeff_b = ''
  end type L2dunits
!
!****

!****d* Datatypes/L2drange
!
! NAME
!    L2drange - A sub-structure of the L2dtype structure, setting valid ranges.
!
! SOURCE
!
  type L2drange
    real(wp), dimension(2) :: level_coeff_a = (/ 0.0_wp, 2000.0_wp/)
    real(wp), dimension(2) :: level_coeff_b = (/ 0.0_wp,    2.0_wp/)
  end type L2drange
!
!****

!****d* Datatypes/L2dtype
!
! NAME
!    L2dtype - A sub-structure of the ROprof structure, containing the defining
!              coefficients for vertical hybrid or eta-type level structures.
!
! SOURCE
!
  type L2dtype
    integer                         :: Npoints       = 0
    character(len = 64)             :: level_type    = "UNKNOWN"
    real(wp), dimension(:), pointer :: level_coeff_a => null() ! Model level coefficients
    real(wp), dimension(:), pointer :: level_coeff_b => null()
    type(L2dunits)                  :: units
    type(L2drange)                  :: range
  end type L2dtype
!
!****

!-------------------------------------------------------------------------------
! 13. Variable lists
!-------------------------------------------------------------------------------

!****id* Datatypes/VlisttypeD0d
!
! NAME
!    VlisttypeD0d - Variable list for scalar double precision variables.
!
! NOTES
!    This parameter is an internal parameter to the ropp_io library and not
!    intended to be used by users.
!
! SOURCE
!
  type VlisttypeD0d
    character(len = 1024)             :: name      = ""
    character(len = 1024)             :: long_name = ""
    character(len = 1024)             :: units     = ""
    real(wp), dimension(2)            :: range     = (/ ropp_io_mdfv, &
                                                        ropp_io_mdfv /)
    real(wp)                          :: data
    type(VlisttypeD0d), pointer       :: next => null()
  end type VlisttypeD0d
!
!****

!****id* Datatypes/VlisttypeD1d
!
! NAME
!    VlisttypeD1d - Variable list for one-dimensional double precision
!                     variables.
!
! NOTES
!    This parameter is an internal parameter to the ropp_io library and not
!    intended to be used by users.
!
! SOURCE
!
  type VlisttypeD1d
    character(len = 1024)             :: name      = ""
    character(len = 1024)             :: long_name = ""
    character(len = 1024)             :: units     = ""
    real(wp), dimension(2)            :: range     = (/ ropp_io_mdfv, &
                                                        ropp_io_mdfv /)
    real(wp), dimension(:), pointer   :: data
    type(VlisttypeD1d), pointer       :: next => null()
  end type VlisttypeD1d
!
!****

!****id* Datatypes/VlisttypeD2d
!
! NAME
!    VlisttypeD2d - Variable list for two-dimensional double precision
!                     variables.
!
! NOTES
!    This parameter is an internal parameter to the ropp_io library and not
!    intended to be used by users.
!
! SOURCE
!
  type VlisttypeD2d
    character(len = 1024)             :: name      = ""
    character(len = 1024)             :: long_name = ""
    character(len = 1024)             :: units     = ""
    real(wp), dimension(2)            :: range     = (/ ropp_io_mdfv, &
                                                        ropp_io_mdfv /)
    real(wp), dimension(:,:), pointer :: data
    type(VlisttypeD2d), pointer       :: next => null()
  end type VlisttypeD2d
!
!****

!****id* Datatypes/Vlisttype
!
! NAME
!    Vlisttype - Variable list for additional variables variables.
!
! NOTES
!    This parameter is an internal parameter to the ropp_io library and not
!    intended to be used by users.
!
! SOURCE
!
  type Vlisttype
    type(VlisttypeD0d), pointer :: VlistD0d => null()
    type(VlisttypeD1d), pointer :: VlistD1d => null()
    type(VlisttypeD2d), pointer :: VlistD2d => null()
  end type Vlisttype
!
!****

!-------------------------------------------------------------------------------
! 14. RO profile
!-------------------------------------------------------------------------------

!****d* Datatypes/ROunits
!
! NAME
!    ROunits - A sub-structure of the ROprof structure, defining (top-level) units.
!
! SOURCE
!
  type ROunits
    character(len = 64) :: pcd          = "bits"
    character(len = 64) :: overall_qual = "percent"
  end type ROunits
!
!****

!****d* Datatypes/ROrange
!
! NAME
!    ROrange - A sub-structure of the ROprof structure, defining (top level) ranges.
!
! SOURCE
!
  type ROrange
    integer,  dimension(2) :: pcd          = (/ 0,      32767      /)
    real(wp), dimension(2) :: overall_qual = (/ 0.0_wp,   100.0_wp /)
  end type ROrange
!
!****

!****d* Datatypes/ROprof
!
! NAME
!    ROprof - Radio Occultation data (profile) data type.
!
! SYNOPSIS
!    use ropp_io_types
!      ...
!    type(ROprof) :: ro_data
!
! NOTES
!    The ROprof structure is composed out of several other structures; see the user guide
!    for a breakdown of the actual element names.
!
! SEE ALSO
!    DT7type
!    GEOtype
!    BGtype
!    L1atype
!    L1btype
!    L2atype
!    L2btype
!    L2ctype
!    L2dtype
!    ROunits
!
! SOURCE
!
  type ROprof
    character(len = 21) :: FmtVersion        = "UNKNOWN" ! File format version ID
    character(len = 40) :: occ_id            = "UNKNOWN" ! Occultation ID
    character(len = 4)  :: leo_id            = "UNKN"    ! LEO identifier
    character(len = 4)  :: gns_id            = "U999"    ! GNSS identifier
    character(len = 4)  :: stn_id            = "UNKN"    ! GSN station identifier
    character(len = 40) :: processing_centre = "UNKNOWN" ! Processing centre
    character(len = 40) :: pod_method        = "UNKNOWN" ! POD processing method
    character(len = 40) :: phase_method      = "UNKNOWN" ! Excess phase processing method
    character(len = 40) :: bangle_method     = "UNKNOWN" ! Bending angle processing method
    character(len = 40) :: refrac_method     = "UNKNOWN" ! Refractivity processing method
    character(len = 40) :: meteo_method      = "UNKNOWN" ! Meteorological processing method
    character(len = 80) :: thin_method       = "UNKNOWN" ! Profile thinning method
    character(len = 40) :: software_version  = "UNKNOWN" ! Software version ID
    type(DT7type)       :: DTocc                         ! Date/time of occultation
    type(DT7type)       :: DTpro                         ! Date/time of processing
    integer             :: PCD          = 65535          ! Product quality flags
    real(wp)            :: overall_qual = ropp_io_mdfv   ! Overall quality value
    type(GEOtype)       :: georef                        ! Georeferencing of the profile
    type(BGtype)        :: bg                            ! Background meta-data
    type(L1atype)       :: Lev1a                         ! Level 1a data
    type(L1btype)       :: Lev1b                         ! Level 1b data
    type(L2atype)       :: Lev2a                         ! Level 2a data
    type(L2btype)       :: Lev2b                         ! Level 2b data
    type(L2ctype)       :: Lev2c                         ! Level 2c data
    type(L2dtype)       :: Lev2d                         ! Level 2d data
    type(ROunits)       :: units                         ! Parameter unit names
    type(ROrange)       :: range                         ! Parameter ranges
    type(Vlisttype)     :: vlist                         ! Additional variables
  end type ROprof

!****

!-------------------------------------------------------------------------------
! 15. Error correlation / covariance matrix
!-------------------------------------------------------------------------------

!****d* Datatypes/ROcorcov
!
! NAME
!    ROcorcov - Error correlation or covariance data type.
!
! SYNOPSIS
!    use ropp_io_types
!      ...
!    type(ROcorcov) :: covar
!
! NOTES
!    The ROcorcov structure contains an error correlation or covariance matrix
!    (the latter will be splitted into an error correlation matrix and an array
!    of diagonal standard deviations). The error correlation matrix is stored in
!    Lapack's packed format for positive definite matrices, i.e. as a 1d array.
!
!    As an additional element, the structure may also contain a latitude range
!    pair which is intended to specify within which latitude band the error
!    covariance is applicable.
!
! SEE ALSO
!
! SOURCE
!
  type ROcorcov
    character(len = 13) :: FmtVersion        = "UNKNOWN" ! File format version ID
    character(len = 40) :: processing_centre = "UNKNOWN" ! Processing centre

    real(wp)                        :: lat_min = -90.0_wp
    real(wp)                        :: lat_max =  90.0_wp
    real(wp), dimension(:), pointer :: sigma   => null()
    real(wp), dimension(:), pointer :: corr    => null()
  end type ROcorcov
!
!****

end module ropp_io_types
