! $Id: gfz2ropp.f90 2105 2009-05-21 16:23:48Z frhl $

!****x* Programs/gfz2ropp *
!
! SYNOPSIS
!   Convert GFZ NRT DAT/DSC files to ROPP netCDF
!
!     > gfz2ropp <gfz-nrt.dat> [-o opfile] [-d] [-h|?] [-v]
!
! ARGUMENTS
!   gfz-nrt.dat   - GFZ 'NRT' DAT (.dat) file containing Level 1b,2a,2b RO
!                   profiles.
!                   May include a path. The companion DSC file containing
!                   meta-data is assumed to be in the same path and have
!                   the same name at the DAT file, but with a type of .dsc.
!                   There is no default for this argument.
!  -o             - specifies the output netCDF file name. The default name
!                   is the same as the DAT/DSC file path/name, but with a
!                   type of .nc
!  -d             - writes additional diagnostic information to stdout
!  -h or ?        - writes usage help information to stdout (and does
!                   nothing more)
!  -v             - write program version to stdout (and does nothing more)
!
! INPUTS
!   Pair of GFZ NRT .dat (profile) and .dsc (meta information)
!
! OUTPUTS
!   ROPP netCDF file - As given with the -o option, or defaults to the
!                      same path and file name as for input DAT file,
!                      but with type .nc
!
! CALLS
!   ropp_io_free
!   ropp_io_init
!   ropp_io_occid
!   ropp_io_write
!   ropp_io_rangecheck
!
! DESCRIPTION
!   Conversion from GFZ native-format GPSRO 'NRT' data (ROPP Levels 1b,2a,2b)
!   to ROPP netCDF format. Input is the pair of DAT (data) & DCS (description)
!   files; the profiles are taken from the former, and meta-data (non-profile
!   header info) from the latter. The data is then written out to a ROPP-
!   standard netCDF file. Only the name of the DAT file is given on the
!   command line; the companion DSC file is assumed to be in the same path and
!   have the same name at the DAT file, but with a type of .dsc.
!
! TODO
!   Incorporate GFZ 'PD' pair of DAT/DSC files so as to include Level 1a
!   (phase/SNR/POD) data into ROPP netCDF.
!
! REFERENCES
!   ROPP Interface File Format. Ref: SAF/GRAS/METO/FMT/ROPP/001
!   ROPP User Guide.            Ref: SAF/GRAS/METO/UG/ROPP/002
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

PROGRAM gfz2ropp

  USE ropp_utils,    ONLY: To_Upper
  USE ropp_io_types, ONLY: ROprof,        &
                           PCD_rising,    &
                           wp
  USE ropp_io,       ONLY: ropp_io_free,  &
                           ropp_io_init,  &
                           ropp_io_occid, &
                           ropp_io_write, &
                           ropp_io_addvar

  IMPLICIT NONE

! Fixed parameters

!   'Version' is the version number of this complete application program

  CHARACTER (LEN=*), PARAMETER :: Version      = "V3.0  1-June-2009"
  CHARACTER (LEN=1), PARAMETER :: SysEndDirChr = "/"
  INTEGER,           PARAMETER :: funit        = 11
  REAL(wp),          PARAMETER :: CtoK         = 273.15_wp
  REAL(wp),          PARAMETER :: KMtoM        = 1000.0_wp
  REAL(wp),          PARAMETER :: SecsPerDay   = 86400.0_wp

! Local variables

  TYPE(ROprof) :: ROdata ! RO main profile structure

  CHARACTER (LEN=256) :: prognam, arg                 ! Program name & command line argument
  CHARACTER (LEN=256) :: datfile, dscfile, opfile     ! I/P & O/P file names
  CHARACTER (LEN=200) :: line                         ! Line read from I/P file
  CHARACTER (LEN=80)  :: key                          ! key word(s) in DSC file
  INTEGER             :: i, j                         ! Loop counters / indices
  INTEGER             :: iarg                         ! Command line argument index
  INTEGER             :: narg                         ! No. of command line arguments
  INTEGER             :: iostatus                     ! I/O Status
  INTEGER             :: NLevs                        ! Number of vertical points in DAT file
  INTEGER             :: SatID                        ! Satellite ID
  INTEGER             :: Year, Month, Day             ! Processing date from DSC file
  INTEGER             :: Hour, Minute, Second         ! Occultation end time from DSC file
  INTEGER             :: SWrev                        ! GFZ revision number
  LOGICAL             :: exists                       ! File exists flag
  LOGICAL             :: setting_occ                  ! .T. if a setting occultation
  LOGICAL             :: DEBUG                        ! Diagnostics dump flag
  REAL(wp)            :: SWver                        ! GFZ POCS software version number
  REAL(wp)            :: sTime, eTime                 ! Start & end time of occultation (secs)

  INTEGER             :: qflag                        ! parameters read but not used in ROPP
  REAL(wp)            :: density, snr1, snr2, basmth  ! parameters read but not used in ROPP
  REAL(wp)            :: alpha, beta, gamma, phaseLC  ! parameters read but not used in ROPP

  LOGICAL             :: lev_1a_exist                 ! Reading Level1a data flag 
  REAL(wp), DIMENSION(:), ALLOCATABLE :: LCF           ! Lost carrier flag from Level 1a file

! Some compilers may need the following declaration to be commented out
  INTEGER             :: IARGC

!-------------------------------------------------------------
! 1. Initialise & parse command line options
!-------------------------------------------------------------

  lev_1a_exist = .false.

  CALL GETARG ( 0, ProgNam )
  i = LEN_TRIM(ProgNam)
  DO WHILE  ( i > 0 .AND. &
              ProgNam(i:i) /= SysEndDirChr )
    i = i - 1
  END DO
  ProgNam = ProgNam(i+1:)
  IF ( ProgNam == " " ) ProgNam = "gfz2ropp"

  datfile = " "       ! no default for i/p file name
  opfile  = " "       ! assume a default generated from i/p file name
  DEBUG   = .FALSE.   ! no diagnostic o/p

  narg      = IARGC()
  iarg      = 1

  DO WHILE ( iarg <= narg )
    CALL GETARG ( iarg, arg )

    SELECT CASE (arg)
      CASE ("-d","-D","--debug")
        DEBUG = .TRUE.

      CASE ("-h","-H","--help","?")
        narg = 0
        datfile = "dummy"

      CASE ("-o","-O","--output")
        iarg = iarg + 1
        CALL GETARG ( iarg, arg )
        opfile = arg

      CASE ("-v","-V","--version")
        WRITE ( *, FMT="(A/)" ) TRIM(ProgNam)//": Version "//TRIM(Version)
        CALL EXIT(0)

      CASE DEFAULT
         datfile = arg

    END SELECT

    iarg = iarg + 1
  END DO

  IF ( datfile == " " ) THEN
    WRITE ( *, FMT="(/A/)" ) " *** No input file(s) specified ***"
    narg = 0
  END IF

  IF ( narg == 0 ) THEN
    WRITE ( *, * ) " "
    WRITE ( *, * ) "Purpose:"
    WRITE ( *, * ) "  Convert a GFZ NRT DAT/DSC pair of RO files to a ROPP netCDF file"
    WRITE ( *, * ) "Usage:"
    WRITE ( *, * ) "> " // TRIM(prognam) // &
                   " ip_file [-o op_file] [-d] [-h|?] [-v]"
    WRITE ( *, * ) "     where ip_file is a GFZ formated NRT .dat file name with an assumed"
    WRITE ( *, * ) "     companion .dsc file with the same name and in the same directory."
    WRITE ( *, * ) "  -o specifies the output (netCDF) file name"
    WRITE ( *, * ) "  -d prints out some additional diagnostics to stdout"
    WRITE ( *, * ) "  -h (or ?) causes only this summary help to be output"
    WRITE ( *, * ) "  -v outputs program version ID (and does nothing else)"
    WRITE ( *, * ) "Defaults:"
    WRITE ( *, * ) "  Input  file name : required"
    WRITE ( *, * ) "  Output file name : from ip_file but .nc"
    WRITE ( *, * ) "See gfz2ropp(1) for details."
    WRITE ( *, * ) " "
    CALL EXIT(0)
  END IF

!-------------------------------------------------------------
! 2. Check GFZ .dat and .dsc input files exist;
!    make output file name if not given on command line
!-------------------------------------------------------------

  INQUIRE ( FILE=datfile, EXIST=exists )
  IF ( .NOT. exists ) THEN
    WRITE ( *, FMT="(A/)" ) "*** GFZ input file " // &
                            TRIM(datfile)       // &
                            " not found"
    CALL EXIT(1)
  ENDIF

  i = INDEX ( datfile, ".dat" )
  IF ( i == 0 ) THEN
    WRITE ( *, FMT="(A/)" ) "*** GFZ input file " // &
                            TRIM(datfile)       // &
                            " not of type '.dat'"
    CALL EXIT(1)
  END IF

  dscfile = datfile(1:i-1)//".dsc"
  INQUIRE ( FILE=dscfile, EXIST=exists )
  IF ( .NOT. exists ) THEN
    WRITE ( *, FMT="(A/)" ) "*** GFZ input file " // &
                            TRIM(dscfile)       // &
                            " not found"
    CALL EXIT(1)
  END IF

  IF ( opfile == " " ) opfile = datfile(1:i-1)//".nc"

!-------------------------------------------------------------
! 3. Read data (.dat) file
!-------------------------------------------------------------

  WRITE ( *, FMT="(A)" ) "Reading " // TRIM(datfile)

  OPEN ( UNIT=funit,   &
         FILE=datfile, &
       STATUS="OLD",   &
       ACTION="READ" )

!-------------------------------------------------------------
! 3.1 Read DAT file header data & extract no. of points in
!     data section plus other meta-info.
!-------------------------------------------------------------

  DO
    READ ( UNIT=funit, FMT="(A)", IOSTAT=iostatus ) line
    IF ( iostatus  /= 0 .OR. &
         line(1:1) /= "#" ) EXIT

    CALL To_Upper ( Line )
    key = line(2:30)
    arg = ADJUSTL(line(31:))

    SELECT CASE (key)
      CASE ("NUMBER OF DATA   LINES")
        READ ( arg, FMT=* ) NLevs

      CASE ("STARTTIME(UTC)") ! Date/Time of start of occultation (& secs since midnight)
        READ ( arg, FMT="(I4,5(1X,I2))" ) ROdata%DTocc%Year,   &
                                          ROdata%DTocc%Month,  &
                                          ROdata%DTocc%Day,    &
                                          ROdata%DTocc%Hour,   &
                                          ROdata%DTocc%Minute, &
                                          ROdata%DTocc%Second
        ROdata%DTocc%Msec = 000
        sTime = ((ROdata%DTocc%Hour*60.0_wp)   &
              +   ROdata%DTocc%Minute)*60.0_wp &
              +   ROdata%DTocc%Second

      CASE ("STARTTIME (UTC)") ! Date/Time of start of occultation (& secs since midnight)
        READ ( arg, FMT="(I4,5(1X,I2))" ) ROdata%DTocc%Year,   &
                                          ROdata%DTocc%Month,  &
                                          ROdata%DTocc%Day,    &
                                          ROdata%DTocc%Hour,   &
                                          ROdata%DTocc%Minute, &
                                          ROdata%DTocc%Second
        ROdata%DTocc%Msec = 000
        sTime = ((ROdata%DTocc%Hour*60.0_wp)   &
              +   ROdata%DTocc%Minute)*60.0_wp &
              +   ROdata%DTocc%Second

      CASE ("ENDTIME(UTC)") ! Date/Time of end of occultation (secs since midnight)
        READ ( arg, FMT="(10X,3(1X,I2))" ) Hour, Minute, Second
        eTime = ((Hour*60.0_wp)   &
              +   Minute)*60.0_wp &
              +   Second

      CASE ("ENDTIME (UTC)") ! Date/Time of end of occultation (secs since midnight)
        READ ( arg, FMT="(10X,3(1X,I2))" ) Hour, Minute, Second
        eTime = ((Hour*60.0_wp)   &
              +   Minute)*60.0_wp &
              +   Second

      CASE ("OCCSAT(PRN)") ! GNSS (GPS) satellite ID
        READ  ( arg, FMT=* ) SatID
        WRITE ( ROdata%GNS_ID, FMT="(A1,I3.3)" ) "G", SatID

      CASE ("OCCULTATION DIRECTION") ! Azimuth angle GNSS-->LEO (deg)
        READ ( arg, FMT=* ) ROdata%GeoRef%Azimuth

      CASE ("ALT_MSL(KM)|LATITUDE(DEG)|LON") 
         lev_1a_exist = .false.
         WRITE(*,fmt="(A)") "Reading Level 1b, 2a, 2b data file"
         
      CASE(" T|SNR_CA|SNR_P1|SNR_P2|X_LEO")
         lev_1a_exist = .true.
         WRITE(*,fmt="(A)") "Reading Level 1a data file"
         NLevs = -1      ! Find number of entries in file
         DO
            READ ( UNIT=funit, FMT="(A)", IOSTAT=iostatus ) line
            IF (iostatus /= 0) THEN
               Exit
            ENDIF
            NLevs = NLevs + 1 
         END DO
         DO i=1,Nlevs+1
            BACKSPACE (UNIT=funit)
         END DO
         
      CASE DEFAULT
    END SELECT
  END DO
  IF ( iostatus == 0 ) BACKSPACE ( UNIT=funit )

!-------------------------------------------------------------
! 3.2 Initialise ROPP structures & fill in static header data
!     not in DAT or DSC files
!-------------------------------------------------------------

  if(lev_1a_exist)then
     CALL ropp_io_init ( ROdata, NLevs, 0, 0, 0, 0, 0 )
     allocate(lcf(NLevs))
  else
     CALL ropp_io_init ( ROdata, 0, NLevs, NLevs, NLevs, 0, 0 )
  endif
     
  ROdata%Processing_Centre = "GFZ GeoForschungsZentrum Potsdam"
  ROdata%Overall_Qual      = 100.0_wp

!-------------------------------------------------------------
! 3.3 Read levels data
!-------------------------------------------------------------

  DO i = 1, NLevs

   if(lev_1a_exist)then
   
    READ ( UNIT=funit, FMT=*, IOSTAT=iostatus ) ROdata%Lev1a%dtime(i),       &
                                                ROdata%Lev1a%snr_L1ca(i),    &
                                                ROdata%Lev1a%snr_L1p(i),     &
                                                ROdata%Lev1a%snr_L2p(i),     &
                                                ROdata%Lev1a%r_leo(i,:),     &
                                                ROdata%Lev1a%v_leo(i,:),     &
                                                ROdata%Lev1a%r_gns(i,:),     &
                                                ROdata%Lev1a%v_gns(i,:),     &
                                                phaseLC,                     &
                                                ROdata%Lev1a%phase_L1(i),    &
                                                ROdata%Lev1a%phase_L2(i),    &
                                                lcf(i)                         
   else

    READ ( UNIT=funit, FMT=*, IOSTAT=iostatus ) ROdata%Lev2a%Alt_Refrac(i),  &
                                                ROdata%Lev1b%Lat_tp(i),      &
                                                ROdata%Lev1b%Lon_tp(i),      &
                                                ROdata%Lev2a%Refrac(i),      &
                                                density,                     &
                                                ROdata%Lev2b%Press(i),       &
                                                ROdata%Lev2b%Temp(i),        &
                                                basmth,                      &
                                                ROdata%Lev1b%Impact(i),      &
                                                alpha, beta, gamma,          &
                                                snr1,  snr2, qflag,          &
                                                ROdata%Lev2a%Geop_Refrac(i), &
                                                ROdata%Lev1b%Bangle(i)
    endif

    IF ( iostatus /= 0 ) EXIT
  END DO

  CLOSE ( UNIT=funit )

  IF ( iostatus > 0 ) THEN
    WRITE ( *, FMT="(A/)" ) "ERROR: I/O error while reading DAT file"
    CALL EXIT(1)
  END IF

!-------------------------------------------------------------
! 3.4 Copy/convert units to ROPP standard
!-------------------------------------------------------------

  if(lev_1a_exist)then

  ROdata%Lev1a%r_leo(:,:)     = ROdata%Lev1a%r_leo(:,:) * KMtoM
  ROdata%Lev1a%v_leo(:,:)     = ROdata%Lev1a%v_leo(:,:) * KMtoM
  ROdata%Lev1a%r_gns(:,:)     = ROdata%Lev1a%r_gns(:,:) * KMtoM
  ROdata%Lev1a%v_gns(:,:)     = ROdata%Lev1a%v_gns(:,:) * KMtoM
  ROdata%Lev1a%phase_qual(:)  = 100.0_wp
  ROdata%Lev1a%reference_frame%r_leo = "ECI"
  ROdata%Lev1a%reference_frame%r_gns = "ECI"

  else

  ROdata%Lev1b%Impact(:)      = ROdata%Lev1b%Impact(:)     * KMtoM
  ROdata%Lev1b%Bangle_Qual(:) = 100.0_wp

  ROdata%Lev2a%Alt_Refrac(:)  = ROdata%Lev2a%Alt_Refrac(:) * KMtoM
  ROdata%Lev2a%Refrac_Qual(:) = 100.0_wp

  ROdata%Lev2b%Geop(:)        = ROdata%Lev2a%Geop_Refrac(:)
  ROdata%Lev2b%Temp(:)        = ROdata%Lev2b%Temp(:)       + CtoK

  endif

!-------------------------------------------------------------
! 3.5 Add lost carrier flag data (Level 1a)
!-------------------------------------------------------------
  
  if(lev_1a_exist)then
  call ropp_io_addvar(ROdata, name = "lcf",             &
                      long_name= "Lost carrier flag",   &
                      units    = " ",                   &
                      range    = (/0.0_wp, 1.0_wp/),    &
                      data     = lcf )
  deallocate(lcf)
  endif

!-------------------------------------------------------------
! 4. Read DSC file
!-------------------------------------------------------------

  WRITE ( *, FMT="(A)" ) "Reading " // TRIM(dscfile)
  OPEN ( UNIT=funit,   &
         FILE=dscfile, &
       STATUS="OLD",   &
       ACTION="READ" )

!-------------------------------------------------------------
! 4.1 Parse lines for key words and extract values for those
!     meta-data that are not in the DAT file
!-------------------------------------------------------------

  setting_occ = .TRUE.

  DO
    READ ( UNIT=funit, FMT="(A)", IOSTAT=iostatus ) line
    IF ( iostatus /= 0 ) EXIT

    CALL To_Upper ( Line )
    i = INDEX ( line, "=" ) + 1
    IF ( i < 3 ) CYCLE
    key = line(1:i-2)

    j = INDEX ( line, ';' ) - 1
    IF ( j < i ) j = LEN_TRIM ( line )
    arg = ADJUSTL(line(i:j))

    SELECT CASE (key)

! DSC file has Processing Date but not the Time. ropp_io_init will have
! intialised Processing Date/Time to current time; if Date in DSC file
! is not today, use it & set Time to the end of the day, else leave
! current time as processing time.

      CASE ("GENERATION DATE") ! Date/Time of processing
        READ ( arg, FMT="(I4,2(1X,I2))" ) Year, Month, Day
        IF ( Year  /= ROdata%DTpro%Year  .OR. &
             Month /= ROdata%DTpro%Month .OR. &
             Day   /= ROdata%DTpro%Day ) THEN
          ROdata%DTpro%Year   = Year
          ROdata%DTpro%Month  = Month
          ROdata%DTpro%Day    = Day
          ROdata%DTpro%Hour   = 23
          ROdata%DTpro%Minute = 59
          ROdata%DTpro%Second = 59
          ROdata%DTpro%Msec   = 999
        ELSE
          ROdata%DTpro%Msec   = 000
        END IF

      CASE ("LOCAL RADIUS OF CURVATURE") ! Earth's radius of curvature at occ. point (m)
        READ ( arg, FMT=* ) ROdata%GeoRef%RoC
        ROdata%GeoRef%RoC = ROdata%GeoRef%RoC * KMtoM

      CASE ("GEOID UNDULATION EGM96")    ! Geoid undulation (EGM96-WGS-84) at occ. point (m)
        READ ( arg, FMT=* ) ROdata%GeoRef%Undulation

!      CASE ("MISSION")  ! LEO Satellite ID
!---------------------------------------------------------------------
! The DSC file seems to have "mission=CHAMP;" even for GRACE-A, so we
! detect the satellite ID from the DAT file name. Remove the marked code
! (and these comments) if/when the DSC file properly states the satellite name.
!-------------------------------------------remove from here --------
        IF ( datfile(8:9) == "CH" ) arg = "CHAMP"
        IF ( datfile(8:9) == "GA" ) arg = "GRACE-A"
        IF ( datfile(8:9) == "GB" ) arg = "GRACE-B"
        IF ( datfile(8:9) == "TS" ) arg = "TERRASAR-X"
        !-------------------------------------------remove to here   --------
        SELECT CASE (arg)
        CASE ("CHAMP")
           ROdata%LEO_ID = "CHMP"
        CASE ("GRACE","GRACE-A")
           ROdata%LEO_ID = "GRAA"
        CASE ("GRACE-B")
           ROdata%LEO_ID = "GRAB"
        CASE ("TERRASAR-X")
           ROdata%LEO_ID = "TSRX"
        CASE DEFAULT
           ROdata%LEO_ID = arg(1:4)
        END SELECT

      CASE ("SETTING(1)/RISING OCCULTATION") ! rising or setting occultation
         IF ( arg /= "1" ) setting_occ = .FALSE.

      !CASE ("SOFTWARE PACKAGE") ! Software ID (major)
      !  READ ( arg(6:), FMT=* ) SWver

      CASE ("REVISION")         ! Software ID (minor)
        READ ( arg, FMT=* ) SWrev

      CASE ("OCCULTATION DIRECTION") ! Azimuth angle GNSS-->LEO (deg)
        READ ( arg, FMT=* ) ROdata%GeoRef%Azimuth

      CASE DEFAULT

    END SELECT

  END DO

  CLOSE ( UNIT=funit )

  IF ( iostatus > 0 ) THEN
    WRITE ( *, FMT="(A/)" ) "ERROR: I/O error while reading DSC file"
    CALL EXIT(1)
  END IF

!-------------------------------------------------------------
! 4.2 Convert GFZ parameters to ROPP standard
!-------------------------------------------------------------

  WRITE ( ROdata%Software_Version, FMT="(F6.3)" ) SWver + SWrev * 0.001_wp
  IF ( SWver < 10.0 ) THEN
    ROdata%Software_Version = "V0"//ADJUSTL(ROdata%Software_Version)
  ELSE
    ROdata%Software_Version = "V" //ADJUSTL(ROdata%Software_Version)
  END IF

! Take nominal lat/lon items from lowest point in profile
! with time offset of zero for rising occultations or end-start
! time for setting. Set PCD rising bit accordingly.

  if (.not. lev_1a_exist) then

  IF ( ROdata%Lev2a%Alt_Refrac(1) < &
       ROdata%Lev2a%Alt_Refrac(NLevs) ) THEN
    ROdata%GeoRef%Lat = ROdata%Lev1b%Lat_tp(1)
    ROdata%GeoRef%Lon = ROdata%Lev1b%Lon_tp(1)
  ELSE
    ROdata%GeoRef%Lat = ROdata%Lev1b%Lat_tp(Nlevs)
    ROdata%GeoRef%Lon = ROdata%Lev1b%Lon_tp(Nlevs)
  END IF

  endif

  ROdata%PCD = 0
  IF ( setting_occ ) THEN
    IF ( eTime < sTime ) eTime = eTime + SecsPerDay  ! in case event spans midnight
    ROdata%GeoRef%Time_Offset  = eTime - sTime
  ELSE
    ROdata%GeoRef%Time_Offset  = 0.0
    ROdata%PCD                 = IBSET(ROdata%PCD,PCD_rising)
  END IF

!-------------------------------------------------------------
! 5. Generate ROPP Occultation ID; optionally dump some key data
!-------------------------------------------------------------

  CALL ropp_io_occid ( ROdata )
  IF ( DEBUG ) THEN
    WRITE ( *, FMT="(A)"       ) " Occultation ID               : "// &
                                 ROdata%Occ_ID
    WRITE ( *, FMT="(A,2F9.3)" ) " Latitude/Longitude           :", &
                                 ROdata%GeoRef%Lat, &
                                 ROdata%GeoRef%Lon
    WRITE ( *, FMT="(A,I6)"    ) " No. of phase/SNR     samples :", &
                                 ROdata%Lev1a%Npoints
    WRITE ( *, FMT="(A,I6)"    ) " No. of bending angle samples :", &
                                 ROdata%Lev1b%Npoints
    WRITE ( *, FMT="(A,I6)"    ) " No. of refractivity  samples :", &
                                 ROdata%Lev2a%Npoints
    WRITE ( *, FMT="(A,I6)"    ) " No. of geophysical   samples :", &
                                 ROdata%Lev2b%Npoints
    WRITE ( *, FMT="(A,I6)"    ) " No. of surface geo.  samples :", &
                                 ROdata%Lev2c%Npoints
    WRITE ( *, FMT="(A,I6)"    ) " No. of model coeff.  levels  :", &
                                 ROdata%Lev2d%Npoints
  ELSE
    i = 1
    WRITE ( *, FMT="(I5,2A,2F8.1)" ) i, " : ",          &
                                     ROdata%occ_id,     &
                                     ROdata%georef%lat, &
                                     ROdata%georef%lon
  END IF

!-------------------------------------------------------------
! 6. Write ROPP netCDF file
!-------------------------------------------------------------

  CALL ropp_io_rangecheck ( ROdata )

  WRITE ( *, FMT="(A)" ) "Writing " // TRIM(opfile)
  CALL ropp_io_write ( ROdata, file=opfile, ierr=iostatus )
  IF ( iostatus > 0 ) THEN
    WRITE ( *, FMT="(A/)" ) "ERROR: I/O error while writing output file"
  END IF

!-------------------------------------------------------------
! 7. Tidy up - deallocate structures & free memory
!-------------------------------------------------------------

  CALL ropp_io_free ( ROdata )

END Program gfz2ropp
