
!

! calculate the bending angle on impact params.

SUBROUTINE alpha_op(nlev,   &   ! no. of model levs (=38)
		    nobs,   &   ! no. of bending angles in profile
		    roc,    &   ! radius of curv. 
		    undul,  &   ! undulation (set to 0.0)
		    lat,    &   ! latitude of ob. location (degrees)
		    pres,   &   ! pressure on mod levels (hPa)
		    temp,   &   ! temp on model levels
		    q,      &   ! specific humidity (g/kg)
	    	    zg,     &   ! geopotential height of model levels (m)
		    a,      &   ! impact parameters
		    alpha)      ! bending angles





		   
IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
INTEGER, INTENT(IN)  :: nobs
REAL,    INTENT(IN)  :: roc
REAL,    INTENT(IN)  :: undul
REAL,    INTENT(IN)  :: lat
REAL,    INTENT(IN)  :: pres(nlev),temp(nlev),q(nlev)
REAL,    INTENT(IN)  :: zg(nlev)
REAL,    INTENT(IN)  :: a(nobs)
REAL,    INTENT(OUT) :: alpha(nobs)
		       
!
! local variables
!
INTEGER :: i
REAL :: refrac(nlev)
REAL :: nr(nlev)
REAL :: roc2


!
! calculate refractivity on model levels
!
          

CALL refrac_levs(nlev,   &
		 pres,   &
		 temp,   &
		 q,      &
		 refrac)

	
!
! calculate the refractive index * radius on model levels
!	  

roc2 = roc + undul
	  
CALL calc_nr(nlev,   &
             roc2,    &
	     lat,    &
	     zg,     &
	     refrac, &
	     nr)

!
! calculate the bending angle for the derived profile from profile
!


CALL calc_alpha(nobs,   &
                 nlev,   &
		 a,      &
		 refrac, &
		 nr,     &
		 alpha)


RETURN
	  
END SUBROUTINE alpha_op


! calculate the refractivity on model levels

SUBROUTINE refrac_levs(nlev,   &		
		       pres,   &
		       temp,   &
		       q,      &
   		       refrac)
		   



USE refrac_info, ONLY:     &
		 Epsilon,  &
	         aval,     &
		 bval,     &
		 RMDI
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
REAL,    INTENT(IN)  :: pres(nlev)   ! in hPa
REAL,    INTENT(IN)  :: temp(nlev),q(nlev)
REAL,    INTENT(OUT) :: refrac(nlev)
		       
!
! local variables
!

INTEGER :: i
REAL :: Ndry,Nhum


!
! calculate the refractivity on  levels
!


refrac(:) = RMDI


DO i = 1, nlev

   Ndry = aval * pres(i)/ temp(i) 
   
   Nhum = 0.0
   
   Nhum = 1.0E-3*bval* pres(i) * q(i)/(Epsilon*temp(i)**2)   


!
! refractivity on ith level
!
       
   refrac(i) = Ndry + Nhum


ENDDO

RETURN
     
     
END SUBROUTINE refrac_levs

! calculate the refractive index radius products

SUBROUTINE calc_nr(nlev,   &
		   roc,    &
		   lat,    &
	    	   zg,     &
   		   refrac, &
		   nr)
		   



USE refrac_info, ONLY:     &
		g,  &
		RMDI
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
REAL,    INTENT(IN)  :: roc
REAL,    INTENT(IN)  :: lat
REAL,    INTENT(IN)  :: zg(nlev)
REAL,    INTENT(IN)  :: refrac(nlev)
REAL,    INTENT(OUT) :: nr(nlev)
		       
!
! local variables
!

INTEGER :: i
REAL :: zed(nlev)
REAL :: rad(nlev)
REAL :: grat
REAL :: radius
REAL :: E_rad
REAL :: g_lat

!
! calculate the radius and g values used in the geopotential/geometric height
! conversion.
!

grat   = g_lat(lat)/g
radius = E_rad(lat)

!
! calculate the geometric heights
!

rad(:) = RMDI
nr(:)  = RMDI 

DO i=1,nlev

   IF (zg(i) > 0.0 .AND. refrac(i) > 0.0 ) THEN

      zed(i)=zg(i)/(grat - zg(i)/radius)

!
! calculate radius value
!
   
      rad(i) = roc + zed(i) 

!
! calculate the radius times refractive index product.
!   

      nr(i) = rad(i) * (1.0+1.0E-6*refrac(i))


   ENDIF


ENDDO

RETURN
          
END SUBROUTINE calc_nr


! calculate the bending angle "alpha" for impact parameters a


SUBROUTINE calc_alpha(nobs,   &
                      nlev,   &
                      a,      &
   		      refrac, &
		      nr,     &
		      alpha)
		   

USE refrac_info, ONLY:     &
  		RMDI,      &
		pi
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nobs           ! size of ob. vector
INTEGER, INTENT(IN)  :: nlev           ! no. of refractivity levels
REAL,    INTENT(IN)  :: a(nobs)        ! impact parameter
REAL,    INTENT(IN)  :: refrac(nlev)   ! refractivity values on levels
REAL,    INTENT(IN)  :: nr(nlev)       ! index * radius product
REAL,    INTENT(OUT) :: alpha(nobs)    ! bending angle 
		       
!
! local variables
!

INTEGER :: i,n,ibot,jbot,kbot
REAL :: kval(nlev-1)
REAL :: root_2pia
REAL :: ref_low
REAL :: nr_low
REAL :: tup,tlow
REAL :: erf 
REAL :: diff_erf
REAL :: dalpha



jbot = 1

DO

  IF (refrac(jbot) > 0.0 .AND. nr(jbot) > 0.0) EXIT
  
  jbot = jbot + 1

ENDDO


kbot = nlev
  
DO i=nlev,jbot+1,-1
 
   IF (nr(kbot) < nr(kbot-1)) EXIT  
   kbot = kbot - 1 

ENDDO

jbot = MAX(jbot,kbot) 


!
! calculate the exponential decay rate between levels
!

DO i=jbot,nlev-1

   kval(i) = LOG(refrac(i)/refrac(i+1)) / &
   MAX(1.0,(nr(i+1)-nr(i)))
   kval(i) = MAX(1.0E-6,kval(i))
   
ENDDO


!
! now calculate the bending angle values
!


alpha(:) = RMDI


DO n=1,nobs

   IF (a(n) < nr(jbot) .OR. a(n) > nr(nlev)) CYCLE    ! bending missing  

   Root_2PIa = SQRT(2.0*pi*a(n))

   ibot = jbot

   DO 

      IF (a(n) < nr(ibot+1)) EXIT 

      ibot=ibot+1

   ENDDO
 
!
!  set bending angle value  
!   

   alpha(n) = 0.0
      
   DO i = ibot, nlev-1
   
      IF ( i == ibot) THEN 
      
         ref_low = refrac(ibot)*EXP(-kval(ibot)*(a(n)-nr(ibot)))
	 nr_low = a(n)
	  
      ELSE 
      
         ref_low = refrac(i) 
	 nr_low  = nr(i) 
	 
      ENDIF

!
! limits used in the error function
!  

      tup = SQRT(kval(i)*(nr(i+1)-a(n)))
      tlow = 0.0                         

      IF (i > ibot) tlow = SQRT(kval(i)*(nr(i)  -a(n)))
      
      
      IF (i < nlev-1) THEN
         		
          diff_erf = erf(tup) - erf(tlow)  				
	
      ELSE

! upper-level, includes the extrapolation to infinity
      
          diff_erf = 1.0 - erf(tlow)	
	
      ENDIF 	
		 	    
      dalpha    =  &
               + 1.0E-6 * Root_2PIa * SQRT(kval(i)) & 
               * ref_low*EXP(kval(i)*(nr_low-a(n)))*diff_erf 
      
      alpha(n) = alpha(n) + dalpha
      
      	  	  
   ENDDO
  
   


ENDDO

RETURN

          
END SUBROUTINE calc_alpha


!
! The error function used in calc_alpha
! 
 
      FUNCTION ERF(X)
      IF (ABS(X).GT.9.0) GO TO 4
      
      IF (X == 0) THEN 
      
         ERF = 0.0
      
      ELSE IF (X > 0.0) THEN
      
         T=1.0/(1.0+0.47047*X)
         ERF=1.0-(0.3480242-(0.0958798-0.7478556*T)*T)*T*EXP(-(X*X))
	 
      ELSE   !(X < 0.0)!
      
        T=1.0/(1.0-0.47047*X)
        ERF=(0.3480242-(0.0958798-0.7478556*T)*T)*T*EXP(-(X*X))-1.0

      ENDIF   
      	 	 
      RETURN
    
    4 ERF=SIGN(1.0,X)
      RETURN
      END

REAL FUNCTION E_rad(lat_in_deg)
!
! calculate the effective radius used to map from geopotential to 
! geometric heights (List,1968),Smithsonian Met Tables.
!
IMPLICIT NONE
! input

REAL lat_in_deg
REAL gval,lat_in_rad,dg_dz

! function
      
REAL g_lat

gval=g_lat(lat_in_deg)
lat_in_rad=1.745329E-2*lat_in_deg

dg_dz= &
3.085462E-6+ &
2.27E-9*COS(2.0*lat_in_rad)- &
2.0E-12*COS(4.0*lat_in_rad)

E_rad=2.0*gval/dg_dz

RETURN
END



REAL FUNCTION g_lat(lat_in_deg)

!
! calculate the value of gravity as a function of latitude
!
IMPLICIT NONE
! input
REAL lat_in_deg   
! local
REAL lat_in_rad,sin_lat,sin_2_lat
!
! calculate the lat in radians
!      

lat_in_rad= 1.745329E-2*lat_in_deg
sin_lat   = SIN(lat_in_rad)
sin_2_lat = SIN(2.0*lat_in_rad)

!
! values taken from List (1968), Smithsonian Met. Tables
!

g_lat=9.780356*(1.0+5.2885E-3*sin_lat**2-5.9E-6*sin_2_lat**2)

RETURN

END



