
!

! calculate the bending angle on impact params.

SUBROUTINE alpha_optl(nlev,   &   ! no. of model levs (=38)
		    nobs,   &   ! no. of bending angles in profile
		    roc,    &   ! radius of curv. 
		    undul,  &   ! undulation (set to 0.0)
		    lat,    &   ! latitude of ob. location (degrees)
		    pres,   &   ! pressure on mod levels (hPa)
		    pres_prime, &
		    temp,   &   ! temp on model levels
		    temp_prime, &
		    q,      &   ! specific humidity (g/kg)
		    q_prime,    &
	    	    zg,     &   ! geopotential height of model levels (m)
		    a,      &   ! impact parameters
		    alpha,  &
		    alpha_prime)      ! bending angles





		   
IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
INTEGER, INTENT(IN)  :: nobs
REAL,    INTENT(IN)  :: roc
REAL,    INTENT(IN)  :: undul
REAL,    INTENT(IN)  :: lat
REAL,    INTENT(IN)  :: pres(nlev),temp(nlev),q(nlev)
REAL,    INTENT(IN)  :: pres_prime(nlev),temp_prime(nlev),q_prime(nlev)
REAL,    INTENT(IN)  :: zg(nlev)
REAL,    INTENT(IN)  :: a(nobs)
REAL,    INTENT(OUT) :: alpha(nobs),alpha_prime(nobs)
		       
!
! local variables
!
INTEGER :: i
REAL :: refrac(nlev)
REAL :: nr(nlev),zg_prime(nlev)
REAL :: refrac_prime(nlev),nr_prime(nlev)
REAL :: roc2

!
! using fixed geopotential heights
!

zg_prime = 0.0

!
! calculate refractivity on model levels
!
          
CALL refrac_levstl(nlev,   &
		 pres,   &
		 pres_prime, &
		 temp,   &
		 temp_prime, &
		 q,      &
		 q_prime, &
		 refrac, &
		 refrac_prime)

	
!
! calculate the refractive index * radius on model levels
!	  

roc2 = roc + undul
	  
CALL calc_nrtl(nlev,   &
             roc2,    &
	     lat,    &
	     zg,     &
	     zg_prime, &
	     refrac, &
	     refrac_prime, &
	     nr, &
	     nr_prime)




!
! calculate the bending angle for the derived profile from profile
!


CALL calc_alphatl(nobs,   &
                 nlev,   &
		 a,      &
		 refrac, &
		 refrac_prime, &
		 nr,     &
		 nr_prime, &
		 alpha,  &
		 alpha_prime)


RETURN
	  
END SUBROUTINE alpha_optl

! calculate the refractivity on observation heights from state vector x


SUBROUTINE refrac_levsTL(nlev,   &
		         pres,  &
			 pres_prime, &
		         temp,      &
		         temp_prime,&
			 q,     &
		         q_prime,   &
			 refrac,    &
		         refrac_prime)
		   


USE refrac_info, ONLY:     &
		 epsilon,&
	         aval,     &
		 bval,     &
		 RMDI
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
REAL,    INTENT(IN)  :: pres(nlev)   ! in hPa
REAL,    INTENT(IN)  :: pres_prime(nlev)   ! in hPa

REAL,    INTENT(IN)  :: temp(nlev)
REAL,    INTENT(IN)  :: temp_prime(nlev)
REAL,    INTENT(IN)  :: q(nlev)
REAL,    INTENT(IN)  :: q_prime(nlev)
REAL,    INTENT(OUT) :: refrac(nlev)
REAL,    INTENT(OUT) :: refrac_prime(nlev)
		       
!
! local variables
!

INTEGER :: i

REAL :: Ndry,Nhum
REAL :: Ndry_prime,Nhum_prime


!
! now calculate the refractivity on pressure levels
!


refrac(:) = RMDI

refrac_prime(:) = 0.0

DO i = 1, nlev

   Ndry = aval * pres(i)/ temp(i) 

   Ndry_prime = Ndry/pres(i)*pres_prime(i) - Ndry/temp(i) * temp_prime(i) 
   
   Nhum = 0.0
   Nhum_prime = 0.0
   
   
   Nhum =  & 
   1.0E-3*bval* pres(i) * q(i)/(Epsilon*temp(i)**2)   

   Nhum_prime = &
   Nhum/pres(i)*pres_prime(i)-2.0*Nhum/temp(i)*temp_prime(i) + Nhum/q(i)*q_prime(i)


!
! refractivity on ith pressure level
!
         
   refrac(i) = Ndry + Nhum

   refrac_prime(i) = Ndry_prime + Nhum_prime 


ENDDO

RETURN

END SUBROUTINE refrac_levsTL

! calculate the nr product

SUBROUTINE calc_nrTL(nlev,         &
		   roc,          &
		   lat,          &
	    	   zg,           &
		   zg_prime,     &
   		   refrac,       &
		   refrac_prime, &
		   nr,           &
		   nr_prime)
		   



USE refrac_info, ONLY:     &
		    g,     &
		    RMDI
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nlev       ! no. of p levels in state vec.
REAL,    INTENT(IN)  :: roc
REAL,    INTENT(IN)  :: lat
REAL,    INTENT(IN)  :: zg(nlev)
REAL,    INTENT(IN)  :: zg_prime(nlev)
REAL,    INTENT(IN)  :: refrac(nlev)
REAL,    INTENT(IN)  :: refrac_prime(nlev)
REAL,    INTENT(OUT) :: nr(nlev)
REAL,    INTENT(OUT) :: nr_prime(nlev)
		       
!
! local variables
!

INTEGER :: i
REAL :: zed(nlev)
REAL :: rad(nlev)
REAL :: zed_prime(nlev)
REAL :: rad_prime(nlev)
REAL :: grat
REAL :: radius
REAL :: E_rad
REAL :: g_lat

!
! calculate the radius and g values used in the geopotential/geometric height
! conversion.
!

grat   = g_lat(lat)/g
radius = E_rad(lat)

!
! calculate the geometric heights
!

rad(:) = RMDI
nr(:)  = RMDI 

rad_prime(:) = 0.0
nr_prime(:) = 0.0


DO i=1,nlev

   IF (zg(i) > 0.0 .AND. refrac(i) > 0.0 ) THEN

      zed(i)=zg(i)/(grat - zg(i)/radius)
      
      zed_prime(i) = grat/(grat-zg(i)/radius)**2 * zg_prime(i)    

!
! calculate radius value
!
   
      rad(i) = roc + zed(i) 

      rad_prime(i) = zed_prime(i) 

!
! calculate the radius times refractive index product.
!   

      nr(i) = rad(i) * (1.0+1.0E-6*refrac(i))

      nr_prime(i) = rad_prime(i) * (1.0+1.0E-6*refrac(i)) &
                  + 1.0E-6* rad(i)* refrac_prime(i)   
          
    

   ENDIF


ENDDO

          
END SUBROUTINE calc_nrTL

! calculate the bending angle "alpha" for impact parameters a

SUBROUTINE calc_alphaTL(nobs,   &
                        nlev,   &
                        a,      &
   		        refrac, &
			refrac_prime, &
		        nr,     &
			nr_prime, &
			alpha, &
		        alpha_prime)
		   

USE refrac_info, ONLY:     &
  		RMDI,      &
		pi
	
		   

IMPLICIT NONE

!
! subroutine args. 
!

INTEGER, INTENT(IN)  :: nobs           ! size of ob. vector
INTEGER, INTENT(IN)  :: nlev           ! no. of refractivity levels
REAL,    INTENT(IN)  :: a(nobs)        ! impact parameter
REAL,    INTENT(IN)  :: refrac(nlev)   ! refractivity values on levels
REAL,    INTENT(IN)  :: refrac_prime(nlev)
REAL,    INTENT(IN)  :: nr(nlev)       ! index * radius product
REAL,    INTENT(IN)  :: nr_prime(nlev)
REAL,    INTENT(OUT) :: alpha(nobs)
REAL,    INTENT(OUT) :: alpha_prime(nobs)    ! bending angle 
		       
!
! local variables
!

INTEGER :: i,n,ibot,jbot
REAL :: kval(nlev-1)
REAL :: kval_prime(nlev-1)
REAL :: root_2pia
REAL :: ref_low
REAL :: ref_low_prime
REAL :: nr_low
REAL :: nr_low_prime
REAL :: tup,tlow
REAL :: tup_prime,tlow_prime
REAL :: dalpha
REAL :: dalpha_prime
REAL :: erf 
REAL :: diff_erf
REAL :: diff_erf_prime


alpha(:)  = RMDI 

alpha_prime(:)  = 0.0 

jbot = 1

DO

  IF (refrac(jbot) > 0.0 .AND. nr(jbot) > 0.0) EXIT
  
  jbot = jbot + 1

ENDDO


!
! calculate the exponential decay rate between levels
!


DO i=jbot,nlev-1

   kval(i) = LOG(refrac(i)/refrac(i+1)) / &
   MAX(1.0,(nr(i+1)-nr(i)))
   
   Kval(i) = MAX(1.0E-6,kval(i)) 
   kval_prime(i) = 0.0
   
   
   IF (kval(i) > 1.0E-6) &
   kval_prime(i) = ((kval(i)*(nr_prime(i)-nr_prime(i+1))) + &
                   (refrac_prime(i)/refrac(i)-              &
		    refrac_prime(i+1)/refrac(i+1)))/        &
                    MAX(1.0,(nr(i+1)-nr(i)))

ENDDO

!
! now calculate the bending angle values
!



DO n=1,nobs


   IF (a(n) < nr(jbot) .OR. a(n) > nr(nlev)) CYCLE  

   Root_2PIa = SQRT(2.0*pi*a(n))

   ibot = jbot

   DO 

      IF (a(n) < nr(ibot+1)) EXIT 

      ibot=ibot+1

   ENDDO
   
!
!  set bending angle value  
!   

   alpha(n) = 0.0
   
   alpha_prime(n) = 0.0
      
   DO i = ibot, nlev-1
   
      IF ( i == ibot) THEN 
      
         ref_low = refrac(i)*EXP(-kval(i)*(a(n)-nr(i)))
	 
	 
	 ref_low_prime = ref_low*               &
	          (refrac_prime(i)/refrac(i) -  &
		   kval_prime(i)*(a(n)-nr(i)) + &
		   kval(i)*nr_prime(i))  
	 
	 
	 nr_low = a(n)
	 
	 nr_low_prime = 0.0


	  
      ELSE 
      
         ref_low = refrac(i) 
	 
	 ref_low_prime = refrac_prime(i)
	 
	 nr_low  = nr(i) 
	 
	 nr_low_prime = nr_prime(i)
	 
      ENDIF
      tup = SQRT(kval(i)*(nr(i+1)-a(n)))

      tup_prime = 0.5*(kval_prime(i)*(nr(i+1)-a(n)) + &
	             kval(i)*nr_prime(i+1))/tup

      tlow = 0.0                         
      
      tlow_prime = 0.0

      IF (i > ibot) THEN
      
          tlow = SQRT(kval(i)*(nr(i)  -a(n)))
	        
          tlow_prime = 0.5*(kval_prime(i)*(nr(i)-a(n)) + &
	               kval(i)*nr_prime(i))/tlow     
      
      ENDIF 
      
      
      IF (i < nlev-1) THEN

      
         diff_erf = erf(tup) - erf(tlow)  				

         diff_erf_prime = 2.0/SQRT(pi)*(EXP(-tup**2)*tup_prime - &
                                      EXP(-tlow**2)*tlow_prime )
  
      ELSE
      
      
         diff_erf = 1.0 - erf(tlow)
          
         diff_erf_prime = -2.0/SQRT(pi)*EXP(-tlow**2)*tlow_prime
       
      ENDIF 
        
		 	    
      dalpha    = &
               + 1.0E-6 * Root_2PIa * SQRT(kval(i)) & 
               * ref_low*EXP(kval(i)*(nr_low-a(n)))*diff_erf 
                           
     
      dalpha_prime = dalpha * (  &
                     ref_low_prime/MAX(1.0E-10,ref_low) + &
                     diff_erf_prime/MAX(1.0E-10,diff_erf) + &
                     kval(i)*nr_low_prime + &
		     (nr_low-a(n) + 0.5/kval(i))* kval_prime(i))              
     
     
      alpha(n) = alpha(n) + dalpha
     
     
     
      alpha_prime(n) = alpha_prime(n) + dalpha_prime
      
      	  	  
   ENDDO


ENDDO

          
END SUBROUTINE calc_alphaTL

