! $Id: ropp_pp_occ_tool.f90 2021 2009-01-16 10:49:04Z frhl $

PROGRAM ropp_pp_occ_tool

!****p* Programs/ropp_pp_occ_tool *
!
! NAME
!    ropp_pp_occ_tool - Pre-processing tool to calculate refractivity profile 
!                       from L1 and L2 excess phase radio occultation data using
!                       geometric optics or wave optics, ionospheric correction,
!                       statistical optimization and Abel transform.
!
! SYNOPSIS
!    ropp_pp_occ_tool <infile(s)> -o <outfile> [-d]
! 
! DESCRIPTION
!    This program reads RO L1 and L2 excess phase data as a function of time 
!    from the input data files and calculates vertical profiles L1 and L2 
!    bending angles, ionospheric corrected bending angle and refractivity. The 
!    result is written to an ROPP formatted output file.
!
! ARGUMENTS
!    <infile(s)>   One (or more) input file names.
!
!    -o <outfile>  Name of output file (default: ropp_pp_occ.nc).
!
! NOTES
!    If the input file is a multifile, or more than one input files are
!    specified, the output file is a multifile.
!
!    Already existing output files will be overwritten.
!
! EXAMPLE
!    To calculate bending angle and refractivity from one of the example 
!    (single-) files in the data directory:
!
!       ropp_pp_occ_tool ../data/input.nc -o example_01.nc
!
!    To calculate bending angle and refractivity profiles from all singlefiles
!    in the data directory:
!
!       ropp_pp_occ_tool ../data/*.nc -o example_02.nc
!
!    Note that the resulting example_02.nc file contains processed data from
!    all example profiles.
!
!    To calculate bending angle and refractivity profiles from
!    all profiles contained in the multifile multi.nc:
!
!       ropp_pp_occ_tool ../data/multi.nc -o example_03.nc
!
!    Since the multi_* file was generated by concatenating the other files
!    in the data directory, example_02.nc and example_03.nc should be identical
!    apart from the file names.
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

!-------------------------------------------------------------------------------
! 1. Declarations
!-------------------------------------------------------------------------------

  USE typesizes, ONLY: wp => EightByteReal
  USE ropp_utils
  USE ropp_io
  USE ropp_io_types, ONLY: ROprof, L1atype, L1btype, L2atype
  USE ropp_pp
  USE ropp_pp_preproc
  USE ropp_pp_types, ONLY: PPConfig, PPDiag

  IMPLICIT NONE

  TYPE(ROprof)    :: ro_data      ! Input RO data 
  TYPE(L1atype)   :: Lev1a        ! Temporary Level1a structure for storage
  TYPE(L1btype)   :: bangle       ! Output bending angle profiles
  TYPE(L1btype)   :: out_ba       ! Corrected bending angles
  TYPE(L1btype)   :: smt_ba       ! Smoothed bending angle
  TYPE(L1btype)   :: mod_ba       ! Model bending angle 
  TYPE(L2atype)   :: out_refrac   ! Retrieved refractivity profile

  TYPE(ppConfig)  :: config       ! Configuration options
  TYPE(ppDiag)    :: diag         ! Diagnostic variables

  INTEGER         :: idummy, imin, imax
  INTEGER         :: i, iargc, argc, k, j
  INTEGER         :: n_files, n_profiles, nbi
  INTEGER         :: ws_go_smooth, ws_go_full, ws_wo, ws_low
  REAL(wp)        :: Pmin, Pmax
  LOGICAL         :: give_help
  LOGICAL         :: earth_ellipsoid = .FALSE.
  LOGICAL         :: output_full = .FALSE.
  LOGICAL         :: twofit = .FALSE.
  LOGICAL         :: ranchk = .TRUE.
  
  REAL(wp), allocatable :: WLC(:)    ! SO-weight of linear combination   !HGL!
  
  CHARACTER(len = 4096), DIMENSION(:), ALLOCATABLE :: ifiles
  CHARACTER(len = 4096)                            :: ofile
  CHARACTER(len = 4096)                            :: cfg_file = " " 
  CHARACTER(len =  256)                            :: buffer
  CHARACTER(len =   64)                            :: version
  CHARACTER(len =   64)                            :: outstr
  CHARACTER(len =   80)                            :: mfile = " "
  CHARACTER(len =   80)                            :: bfile = " "
  CHARACTER(len =   80)                            :: navfile = " "
  CHARACTER(len =   10)                            :: method = " "
  CHARACTER(len =   10)                            :: occmethod = " "
  CHARACTER(len =   10)                            :: filtmethod = " "
  CHARACTER(len =    4)                            :: istr
  CHARACTER(len =   10)                            :: nstr
  CHARACTER(len =   10)                            :: pstr1
  CHARACTER(len =   10)                            :: pstr2

!-------------------------------------------------------------------------------
! 2. Default settings
!-------------------------------------------------------------------------------

  version   = ADJUSTL("5.0")
  give_help = .FALSE.
  ofile     = "ropp_pp_occ.nc"

  CALL message(msg_noin, '')
  CALL message(msg_noin, &
       '----------------------------------------------------------------------')
  CALL message(msg_noin, &
       ' ROPP Occultation Pre-processor Tool - v' // TRIM(version))
  CALL message(msg_noin, &
       '----------------------------------------------------------------------')
   CALL message(msg_noin, '')

!-------------------------------------------------------------------------------
! 3. Command line arguments
!-------------------------------------------------------------------------------

  argc = iargc()
  i = 1 
  n_files = 0
  ALLOCATE(ifiles(argc))

  DO WHILE(i <= argc)
     CALL getarg(i, buffer)
     SELECT CASE (buffer)
        CASE('-o')                          ! Output file name (netCDF output)
           CALL getarg(i+1, buffer)
           ofile = buffer
           i = i + 1
        CASE('-c')                          ! Configuration file name
           CALL getarg(i+1, buffer)
           cfg_file = buffer
           i = i + 1
        CASE('-m')                          ! Ionospheric correction method
           CALL getarg(i+1, buffer)
           method = buffer
           i = i + 1
        CASE('-mfile')                      ! Model refractivity file
           CALL getarg(i+1, buffer)
           mfile = buffer
           i = i + 1  
        CASE('-bfile')                    ! Background atmosphere profile file
           CALL getarg(i+1, buffer)
           bfile = buffer
           i = i + 1  
        CASE('-navfile')                    ! External navigation bit file
           CALL getarg(i+1, buffer)
           navfile = buffer
           i = i + 1    
        CASE('-occ', '--occ')               ! Occultation processing method
          CALL getarg(i+1, buffer)
          occmethod = buffer
          i = i + 1
        CASE('-filter', '--filter')         ! Filtering method
          CALL getarg(i+1, buffer)
          filtmethod = buffer
          i = i + 1
        CASE('-fit', '--fit')               ! 2 parameter fitting method
          twofit = .true.
        CASE('-ellipsoid')                  ! Output height wrt reference ellipsoid
          earth_ellipsoid = .TRUE.
        CASE('-full')                       ! Output full height grid (IC plus MSIS) 
          output_full = .TRUE.  
        CASE('-d')                          ! 'Diagnostic' output mode
          msg_MODE = VerboseMode
        CASE ('-no_ranchk')                 ! Use no rangecheck on output
           ranchk = .FALSE.
        CASE('-h', '-help', '--help')       ! Give some help
           give_help = .TRUE.
        CASE('-V', '-version', '--version') ! Give some version information
           CALL version_info(version)
           CALL EXIT(0)
        CASE default                        ! Input file name
           n_files = n_files + 1
           ifiles(n_files) = buffer
     END SELECT
     i = i + 1
  END DO

  IF (argc == 0 .OR. n_files == 0 .OR. give_help) THEN
     CALL usage()
     CALL EXIT(0)
  ENDIF

  ! 3.1 Read configuration file (if exists), preserving command-line options

  IF (method /= " ")    config%method = method
  IF (mfile /= " ")     config%mfile = mfile
  IF (bfile /= " ")     config%bfile = bfile
  IF (navfile /= " ")   config%navbit_file = navfile
  IF (occmethod /= " ") config%occ_method = occmethod
  IF (filtmethod /= " ") config%filter_method = filtmethod

  IF (cfg_file /= " ") THEN
     CALL message(msg_info, &
          "Reading configuration file " // TRIM(cfg_file) // ".\n")
     
     CALL ropp_pp_read_config(cfg_file, config)
     IF (method /= " ")     config%method = method
     IF (mfile /= " ")      config%mfile = mfile
     IF (bfile /= " ")      config%bfile = bfile
     IF (navfile /= " ")    config%navbit_file = navfile
     IF (occmethod /= " ")  config%occ_method = occmethod
     IF (filtmethod /= " ") config%filter_method = filtmethod
     IF (twofit)            config%nparm_fit = 2
  ENDIF
 
!-------------------------------------------------------------------------------
! 4. Remove pre-existing output file    
!-------------------------------------------------------------------------------

  CALL file_delete(ofile, idummy)

!-------------------------------------------------------------------------------
! 5. Loop over all input files and profiles
!-------------------------------------------------------------------------------

  DO k = 1, n_files

     n_profiles = ropp_io_nrec(ifiles(k))

     DO i = 1, n_profiles
         
        WRITE(istr, '(i4)') i
        WRITE(nstr, '(i6)') n_profiles
        CALL message(msg_info, "Processing profile " // istr // " of " // nstr )
        
!-------------------------------------------------------------------------------
! 6. Read data
!-------------------------------------------------------------------------------
        
! 6.1 Read input file

        CALL message(msg_info, &
          "Reading input data file " // TRIM(ifiles(k)) // ".\n")
        CALL ropp_io_read(ro_data, ifiles(k), rec = i, ranchk = .TRUE.)
        CALL message(msg_info, "(" // TRIM(ro_data%occ_id) // ") \n")

        IF (ro_data%Lev1a%Npoints == 0) THEN
          config%obs_ok = .FALSE.
          CALL ropp_io_free(ro_data%vlist) 
          CALL ropp_io_write(ro_data, ofile, append = .TRUE., ranchk = ranchk )
          CALL message(msg_fatal, "FAILURE: No Level1a data in file " //    &
                                   ifiles(k) //  "\n")
        ENDIF
        
! 6.2 Shrink ro_data to correct size (for multiple profiles)

        CALL ropp_io_init(Lev1a, ro_data%lev1a%npoints)
        Lev1a = ro_data%lev1a 
        ro_data%lev1a = Lev1a
        CALL ropp_io_free(Lev1a)
        CALL ropp_io_free(ro_data%Lev1b)

!-------------------------------------------------------------------------------
! 7. Check coordinate frames 
!-------------------------------------------------------------------------------        

        CALL ropp_pp_set_coordinates(ro_data)

!-------------------------------------------------------------------------------
! 8. Determine occultation point georeference information
!-------------------------------------------------------------------------------
        
        CALL occ_point( ro_data%lev1a%r_leo,  ro_data%lev1a%r_gns,       &
                        ro_data%georef%lat,   ro_data%georef%lon,        &
                        ro_data%georef%r_coc, ro_data%georef%roc,        &
                        ro_data%georef%azimuth,                          &
                        ro_data%georef%undulation,                       &
                        config%egm96, config%corr_egm96)
        WRITE(outstr,'(2(A,1x,f7.2))') &
           "Occultation point: Lat= ", ro_data%georef%lat, &
           ", Lon= ", ro_data%georef%lon
        CALL message(msg_diag, outstr)

!-------------------------------------------------------------------------------
! 9. Mission-specific pre-processing and input data cut-off
!-------------------------------------------------------------------------------

        CALL ropp_pp_preprocess(ro_data, config, diag)

!-------------------------------------------------------------------------------
! 10. Initialise bending angle structures
!-------------------------------------------------------------------------------

        IF (config%obs_ok) THEN

          CALL ropp_io_init(smt_ba, ro_data%Lev1a%Npoints)
          CALL ropp_io_init(bangle, ro_data%Lev1a%Npoints)
        
          CALL tangent_point(ro_data%lev1a%r_leo,  ro_data%lev1a%r_gns,       &
                             bangle%lat_tp, bangle%lon_tp, bangle%azimuth_tp)

!-------------------------------------------------------------------------------
! 11. Geometric optics processing
!-------------------------------------------------------------------------------
        
          CALL message(msg_info, &
             "Retrieving bending angle profiles by GEOMETRIC OPTICS \n") 
          

! 11.1 Calculate smoothed bending angle
           
          DO j=1,ro_data%Lev1a%Npoints
            smt_ba%impact(j) =                                                 &
               impact_parameter(ro_data%Lev1a%r_leo(j,:)-ro_data%georef%r_coc, &
                                ro_data%Lev1a%r_gns(j,:)-ro_data%georef%r_coc )
          ENDDO
          Pmax = MAXVAL(smt_ba%impact)
          Pmin = MAX(MINVAL(smt_ba%impact), ro_data%georef%roc+2000.0_wp)
          ws_go_smooth = CEILING( config%fw_go_smooth *                        &
                                    (ro_data%Lev1a%Npoints-1)/ABS(Pmax-Pmin))
        
          WRITE(pstr1, '(F10.3)') Pmin-ro_data%georef%roc
          WRITE(pstr2, '(F10.3)') Pmax-ro_data%georef%roc
          CALL message(msg_diag, "Smoothed bending angle. ")
          CALL message(msg_diag, "Pmin = " // pstr1 // " Pmax = " // pstr2)
          WRITE(nstr, '(i8)') ws_go_smooth
          CALL message(msg_diag, "ws_go_smooth " // nstr)
          
          IF (Pmax > Pmin) THEN

            CALL ropp_pp_bending_angle_go( ro_data%Lev1a%dtime,      &
                                           ro_data%Lev1a%r_leo,      &
                                           ro_data%Lev1a%r_gns,      &
                                           ro_data%georef%r_coc,     &       
                                           ro_data%Lev1a%phase_L1,   &
                                           ro_data%Lev1a%phase_L2,   &
                                           ws_go_smooth,             &
                                           config%filter_method,     &
                                           smt_ba%impact_L1,         &
                                           smt_ba%bangle_L1,         &
                                           smt_ba%impact_L2,         &
                                           smt_ba%bangle_L2  )


            CALL ropp_pp_linear_combination( smt_ba%impact_L1,     &
                                             smt_ba%bangle_L1,     &
                                             smt_ba%impact_L2,     &
                                             smt_ba%bangle_L2,     &
                                             smt_ba%impact,        &
                                             smt_ba%bangle )

          ELSE
             
            config%obs_ok = .FALSE.
            CALL message(msg_warn, "Cannot process profile (smt): Pmin > Pmax")
            
          ENDIF
  
! 11.2 Calculate bending angle
        
          bangle%impact = smt_ba%impact_L1
          Pmax = MAXVAL(smt_ba%impact_L1)
          Pmin = MAX(MINVAL(smt_ba%impact_L1), ro_data%georef%roc+2000.0_wp)
          ws_go_full = CEILING(config%fw_go_full*(ro_data%Lev1a%Npoints - 1)/  &
                         ABS(Pmax - Pmin)) 
          
          WRITE(pstr1, '(F10.3)') Pmin-ro_data%georef%roc
          WRITE(pstr2, '(F10.3)') Pmax-ro_data%georef%roc
          CALL message(msg_diag, "Full resolution bending angle. ")
          CALL message(msg_diag, "Pmin = " // pstr1 // " Pmax = " // pstr2)
          WRITE(nstr, '(i8)') ws_go_full
          CALL message(msg_diag, "ws_go_full " // nstr)

          IF (Pmax > Pmin) THEN

            CALL ropp_pp_bending_angle_go(ro_data%Lev1a%dtime,    &
                                          ro_data%Lev1a%r_leo,    &
                                          ro_data%Lev1a%r_gns,    &
                                          ro_data%georef%r_coc,   &       
                                          ro_data%Lev1a%phase_L1, &
                                          ro_data%Lev1a%phase_L2, &
                                          ws_go_full,             &
                                          config%filter_method,   &
                                          bangle%impact_L1,       &
                                          bangle%bangle_L1,       &
                                          bangle%impact_L2,       &
                                          bangle%bangle_L2 )

            WRITE(nstr, '(i10)') ro_data%Lev1a%Npoints
            WRITE(pstr2, '(f6.1)') Pmax/1000.0_wp
            WRITE(pstr1, '(f6.1)') Pmin/1000.0_wp
            CALL message(msg_info,                                             &
                           TRIM(nstr) // " data points in output between " //  &
                           TRIM(pstr1) // "km and " // TRIM(pstr2) // "km \n")

          ELSE

            config%obs_ok = .FALSE.
            CALL message(msg_warn, "Cannot process profile (full): Pmin > Pmax")

          ENDIF
         
!-------------------------------------------------------------------------------
! 12. Wave optics processing
!-------------------------------------------------------------------------------

          IF ( INDEX(config%occ_method, "WO" ) == 1 ) THEN
        
            CALL message(msg_info, &
                    "Retrieving bending angle profiles by WAVE OPTICS \n")

            ws_wo = CEILING(config%fw_wo * (bangle%npoints-1.0_wp) /       &
                       ABS(Pmax - Pmin))
            ws_low = CEILING(config%fw_low * (bangle%npoints-1.0_wp) /     &
                       ABS(Pmax - Pmin))

            WRITE(nstr,'(I5)') ws_go_smooth
            CALL message(msg_diag, 'WM = ' // nstr)           
            WRITE(nstr,'(I5)') ws_wo
            CALL message(msg_diag, 'W = ' // nstr)
            WRITE(nstr,'(I5)') ws_low
            CALL message(msg_diag, 'WL = ' // nstr)
                        
            IF (Pmax > Pmin) THEN

              CALL ropp_pp_bending_angle_wo( ro_data%Lev1a%dtime,      &
                                             ro_data%Lev1a%r_leo,      &
                                             ro_data%Lev1a%r_gns,      &
                                             ro_data%georef%r_coc,     &
                                             ro_data%georef%roc,       & 
                                             ro_data%Lev1a%phase_L1,   &
                                             ro_data%Lev1a%phase_L2,   &
                                             ro_data%Lev1a%snr_L1ca,   &
                                             ro_data%Lev1a%snr_L2p,    &
                                             ws_go_smooth,             &
                                             ws_wo,                    &
                                             ws_low,                   &
                                             config%hmax_wo,           &
                                             config%filter_method,     &
                                             config%opt_DL2,           &
                                             config%cff,               &
                                             config%dsh,               &
                                             bangle%impact_L1,         &
                                             bangle%bangle_L1,         &
                                             bangle%bangle_L1_sigma,   &
                                             bangle%impact_L2,         &
                                             bangle%bangle_L2,         &
                                             bangle%bangle_L2_sigma,   &
                                             diag )

              WRITE(nstr, '(i10)') ro_data%Lev1a%Npoints
              WRITE(pstr2, '(f6.1)') Pmax/1000.0_wp
              WRITE(pstr1, '(f6.1)') Pmin/1000.0_wp
              CALL message(msg_info,                                 &
                 TRIM(nstr) // " data points in output between " //  &
                 TRIM(pstr1) // "km and " // TRIM(pstr2) // "km \n")
              
            ELSE

              config%obs_ok = .FALSE.
              CALL message(msg_warn, "Cannot process profile (wo): Pmin > Pmax")
           
            ENDIF


          ENDIF

!-------------------------------------------------------------------------------
! 13. Check that profiles are monotonously increasing in height
!     (index 1 towards surface) and valid bending angles
!-------------------------------------------------------------------------------

          CALL ropp_pp_monotonous(bangle%impact_L1, -1)
          CALL ropp_pp_monotonous(bangle%impact_L2, -1)
          CALL ropp_pp_monotonous(smt_ba%impact, -1)

          IF ( ANY(bangle%bangle_L1 <= ropp_MDTV) .OR.      &
             ANY(bangle%bangle_L2 <= ropp_MDTV) ) THEN
            config%obs_ok = .FALSE.
            CALL message(msg_warn, "Cannot process profile: " //    &
                                   "Invalid bending angles computed \n")
          ENDIF

        ENDIF

!-------------------------------------------------------------------------------
! 14. Compute un-optimised ionospheric corrected bending angle profile (LC)
!-------------------------------------------------------------------------------
        
        IF (config%obs_ok) THEN

          CALL ropp_pp_linear_combination( bangle%impact_L1, bangle%bangle_L1, &
                                           bangle%impact_L2, bangle%bangle_L2, &
                                           bangle%impact,    bangle%bangle )
                            
!-------------------------------------------------------------------------------
! 15. Define configuration parameters
!-------------------------------------------------------------------------------
               
          config%r_curve = ro_data%georef%roc 
          config%npoints = bangle%npoints
          
          config%Pmax = MAXVAL(bangle%impact_L1(:))  
          config%Pmin = MINVAL(bangle%impact_L1(:))
          
!-------------------------------------------------------------------------------
! 16. Interpolate input data to standard grid
!-------------------------------------------------------------------------------

 ! 16.1 Calculate size of standard grid

          IF ( INDEX(config%method, "NONE" ) == 1 ) THEN
            Pmax = MAXVAL(bangle%impact_L1(:)) - 5000.0_wp
          ELSE
            Pmax = config%ztop_invert + config%r_curve
          ENDIF
          Pmin = MINVAL(bangle%impact_L1(:))
          nbi = 1 + CEILING((Pmax - Pmin)/config%dpi)

          WRITE(pstr2, '(f10.3)') Pmax-config%r_curve
          WRITE(pstr1, '(f10.3)') Pmin-config%r_curve
          CALL message(msg_diag, "Pmin = " // pstr1 // " Pmax = " // pstr2)
          WRITE(nstr, '(I6)') nbi
          CALL message(msg_diag, "Standard grid size nbi = " // nstr)

! 16.2 Initialise standard grid data structures for output
     
          CALL ropp_io_init(out_ba, nbi)
          CALL ropp_io_init(out_refrac, nbi)
          
! 16.3 Interpolate input data onto standard grid

          CALL ropp_pp_merge_profile(bangle%impact_L1, bangle%bangle_L1, &
                                     bangle%impact_L2, bangle%bangle_L2, &
                                     out_ba%impact_L1, out_ba%bangle_L1, &
                                     out_ba%impact_L2, out_ba%bangle_L2, &
                                     Pmin, Pmax)

          CALL ropp_pp_merge_profile(bangle%impact_L1,bangle%bangle_L1_sigma, &
                                     bangle%impact_L2,bangle%bangle_L2_sigma, &
                                     out_ba%impact_L1,out_ba%bangle_L1_sigma, &
                                     out_ba%impact_L2,out_ba%bangle_L2_sigma, &
                                     Pmin, Pmax)        

          ALLOCATE(WLC(nbi))   !HGL!
          WLC(:) = 1.0         !HGL!            


!-------------------------------------------------------------------------------
! 17. Ionospheric correction of bending angle profile by linear combination 
!-------------------------------------------------------------------------------

          IF ( INDEX(config%method, "NONE" ) == 1 ) THEN
                    
            CALL message(msg_info,   &
               "Correcting bending angle profile by LINEAR COMBINATION \n")
            ro_data%bangle_method = TRIM(config%occ_method) // "(LC)"
      
            CALL ropp_pp_linear_combination( out_ba%impact_L1,     &
                                             out_ba%bangle_L1,     &
                                             out_ba%impact_L2,     &
                                             out_ba%bangle_L2,     &
                                             out_ba%impact_opt,    &
                                             out_ba%bangle_opt )
            
            imax = SUM(MINLOC(ABS(out_ba%impact_L1(:)-(config%Pmax-5000.0))))
            imin = SUM(MINLOC(ABS(out_ba%impact_L1(:)-config%Pmin)))

!-------------------------------------------------------------------------------
! 18. Ionospheric correct bending angle profile by statistical optimization
!-------------------------------------------------------------------------------

          ELSE
         
            CALL message(msg_info,   &
              "Correcting bending angle profile by STATISTICAL OPTIMISATION \n")
            ro_data%bangle_method = TRIM(config%occ_method) // " (StatOpt)"
              
! 18.1 Retrieve model bending angles on L1 impact parameter levels

            CALL ropp_io_init(mod_ba, nbi)

            IF (INDEX(config%method, "GMSIS" ) == 1 ) THEN
              CALL ropp_pp_search_model_refraction( config%mfile, &
                                           ro_data%dtocc%month,   &
                                           ro_data%georef%lat,    &
                                           ro_data%georef%lon,    &
                                           smt_ba%impact,         &
                                           smt_ba%bangle,         &
                                           out_ba%impact_L1,      &
                                           mod_ba%bangle,         &
                                           config )
              
            ELSE IF (INDEX(config%method, "MSIS" ) == 1 ) THEN
              CALL ropp_pp_model_refraction( config%mfile,          &
                                           ro_data%dtocc%month,   &
                                           ro_data%georef%lat,    &
                                           ro_data%georef%lon,    &
                                           out_ba%impact_L1,      &
                                           mod_ba%bangle,         &
                                           config )


            ELSE IF (INDEX(config%method, "BG" ) == 1 ) THEN
              CALL ropp_pp_bg_refraction( config%bfile,          &
                                          ro_data%dtocc%month,   &
                                          ro_data%georef%lat,    &
                                          ro_data%georef%lon,    &
                                          out_ba%impact_L1,      &
                                          mod_ba%bangle,         &
                                          config )

            ELSE
              config%obs_ok = .FALSE.
              CALL message(msg_warn,    &
                 "Statistical optimisation method " // config%method // &
                 " not supported")
              EXIT
            ENDIF

! 18.2 Fit model bending angle with (smoothed) observed bending angles
            
            CALL ropp_pp_fit_model_refraction( smt_ba%impact,     &
                                               smt_ba%bangle,     &
                                               out_ba%impact_L1,  &
                                               mod_ba%bangle,     &
                                               config )
            
! 18.3 Perform ionospheric correction with statistical optimization

            imax = SUM(MINLOC(ABS(out_ba%impact_L1(:)-(config%Pmax-5000.0))))
            imin = SUM(MINLOC(ABS(out_ba%impact_L1(:)-config%Pmin)))
            
            WLC(:) = 0.0         !HGL!
                  
            ! Merge observed bending angles with model above observation top
            WHERE ( out_ba%impact_L1 > config%Pmax - 5000.0_wp ) 
              out_ba%bangle_L1(:) = mod_ba%bangle(:)
              out_ba%bangle_L2(:) = mod_ba%bangle(:)
              out_ba%bangle_L1_sigma(:) = 0.0_wp
              out_ba%bangle_L2_sigma(:) = 0.0_wp
            END WHERE

            IF (imin < imax) THEN
              out_ba%impact_opt(:) = out_ba%impact_L1(:)
              out_ba%bangle_opt(:) = out_ba%bangle_L1(:)
              CALL ropp_pp_ionospheric_correction(           &
                             out_ba%impact_L1(imin:imax),    &  ! L1 impact
                             out_ba%bangle_L1(imin:imax),    &  ! L1 bending
                             out_ba%impact_L2(imin:imax),    &  ! L2 impact
                             out_ba%bangle_L2(imin:imax),    &  ! L2 bending
                             out_ba%impact_L1(imin:imax),    &  ! Model impact
                             mod_ba%bangle(imin:imax),       &  ! Model bending
                             config,                         &  ! Configuration
                             out_ba%impact_opt(imin:imax),   &  ! Opt impact
                             out_ba%bangle_opt(imin:imax),   &  ! Opt bending
                             diag,                           &  ! Diagnostics                                     !HGL!
                             WLC(imin:imax))                    ! Weight of the linear combination of L1 and L2   !HGL!
              diag%err_neut = diag%err_neut +    &
                                 out_ba%bangle_L1_sigma(imin:imax)**2
              diag%sq = 100.0_wp *   &
                    MAXVAL(SQRT(diag%err_neut(:))/out_ba%bangle_opt(imin:imax))
            ELSE
              config%obs_ok = .FALSE.
              CALL message(msg_warn,   &
                   "Cannot perform ionospheric correction: Imin >= Imax \n")
            ENDIF
            
          ENDIF
        
! 18.4 Interpolate LC bending angles, lat_tp, lon_tp, azimuth_tp to output grid

          out_ba%impact = out_ba%impact_opt
          CALL ropp_pp_interpol(bangle%impact, out_ba%impact, &
                              bangle%bangle, out_ba%bangle)
          CALL ropp_pp_interpol(bangle%impact, out_ba%impact, &
                              bangle%lat_tp, out_ba%lat_tp)
          CALL ropp_pp_interpol(bangle%impact, out_ba%impact, &
                              bangle%lon_tp, out_ba%lon_tp)
          CALL ropp_pp_interpol(bangle%impact, out_ba%impact, &
                              bangle%azimuth_tp, out_ba%azimuth_tp)
          
          WRITE(nstr, '(i10)') imax-imin+1
          WRITE(pstr2, '(f6.1)') out_ba%impact(imax)/1000.0_wp
          WRITE(pstr1, '(f6.1)') out_ba%impact(imin)/1000.0_wp
          CALL message(msg_info,                                 &
             TRIM(nstr) // " data points in output between " //  &
             TRIM(pstr1) // "km and " // TRIM(pstr2) // "km \n")
          
        END IF
        
!-------------------------------------------------------------------------------
! 19. Perform inverse Abel transform to compute refractivity
!-------------------------------------------------------------------------------

        IF (config%obs_ok) THEN
                    
          ro_data%refrac_method = "ABEL (" // TRIM(config%abel) // ")"

! 19.1 Abel inversion of corrected bending angle profile

          IF ( INDEX(config%method, "NONE" ) == 1 ) THEN
                          
            CALL message(msg_info,   &
              "Retrieving refractivity profile by " // TRIM(config%abel) //  &
              " ABEL TRANSFORM \n") 
            
            IF ( INDEX(config%abel, "LIN" ) == 1 ) THEN
              CALL ropp_pp_invert_LIN(out_ba%impact_opt, out_ba%bangle_opt,    &
                                      out_ba%impact_opt, out_refrac%refrac)
            ELSE IF ( INDEX(config%abel, "EXP" ) == 1 ) THEN
              CALL ropp_pp_invert_EXP(out_ba%impact_opt, out_ba%bangle_opt,    &
                                      out_ba%impact_opt, out_refrac%refrac)
            ELSE
              config%obs_ok = .FALSE.
              CALL message(msg_warn, &
                 "Abel integral method " // config%abel // " not supported")
            ENDIF

! 19.2 Abel inversion of corrected bending angle profile with stat opt
              
          ELSE
            
             CALL message(msg_info,   &
              "Retrieving refractivity profile by " // TRIM(config%abel) //  &
              " ABEL TRANSFORM with STAT OPT \n") 

            CALL ropp_pp_invert_refraction( config%mfile,            &
                                            ro_data%dtocc%month,     &
                                            ro_data%georef%lat,      &
                                            ro_data%georef%lon,      &
                                            out_ba%impact_opt,       &
                                            out_ba%bangle_opt,       &
                                            out_refrac%geop_refrac,  &
                                            out_refrac%refrac,       &
                                            config )
            
          ENDIF
                                            
!-------------------------------------------------------------------------------
! 20. Compute output height scales
!-------------------------------------------------------------------------------

          ! Override default output height scales wrt geoid if requested
          IF (earth_ellipsoid) ro_data%georef%undulation = ropp_MDFV

          IF (ro_data%georef%undulation > ropp_MDTV) THEN
            CALL message(msg_info, "Writing output altitude scales " // &
               "with respect to EGM96 GEOID")
            out_refrac%alt_refrac =                                           &
               ((out_ba%impact_opt / (1.0_wp + out_refrac%refrac * 1.e-6_wp)) &
               - (ro_data%GEOref%roc + ro_data%GEOref%undulation)) 
          ELSE
            IF (.not. earth_ellipsoid)   &
               CALL message(msg_warn, "Invalid undulation calculated. " //   &
             "Check for valid EGM geoid coefficient and correction file.") 
            CALL message(msg_info, "Writing output altitude scales " //    &
               "with respect to WGS84 ELLIPSOID.")
            out_refrac%alt_refrac =                                           &
               ((out_ba%impact_opt / (1.0_wp + out_refrac%refrac * 1.e-6_wp)) &
               - ro_data%GEOref%roc) 
          ENDIF
          out_refrac%geop_refrac =                                        &
              geometric2geopotential(ro_data%georef%lat, out_refrac%alt_refrac)
          
!-------------------------------------------------------------------------------
! 21. Copy retrieved profiles to RO structure 
!       - only output data within observed range
!-------------------------------------------------------------------------------
          
          ! Override default output only within observed range if requested
          IF (.not. output_full) THEN     
            out_ba%npoints = Imax - Imin + 1 
            out_refrac%npoints = Imax - Imin + 1
          ENDIF
          
          CALL ropp_io_roprof2roprof(out_ba, ro_data%lev1b)
          CALL ropp_io_roprof2roprof(out_refrac, ro_data%lev2a)


!-------------------------------------------------------------------------------
! 22. Compute Tdry and Pdry
!-------------------------------------------------------------------------------

          IF (config%output_tdry) THEN
            CALL message(msg_info, "Computing dry temperature and pressure \n")
            CALL ropp_io_init(ro_data%lev2b, ro_data%lev2a%npoints)
            ro_data%lev2b%geop = ro_data%lev2a%geop_refrac
            ro_data%lev2b%shum = 0.0_wp
            CALL ropp_pp_tdry(ro_data%georef%lat, ro_data%lev2a%alt_refrac,   &
             ro_data%lev2a%refrac, ro_data%lev2b%shum, ro_data%lev2b%temp,    &
             ro_data%lev2b%press, config%ztop_invert)
            ro_data%lev2b%press = ro_data%lev2b%press / 100.0_wp ! convert to hPa
          ENDIF
        ENDIF

!-------------------------------------------------------------------------------
! 23. Write data
!-------------------------------------------------------------------------------

        CALL ropp_io_free(ro_data%vlist)   !! interim, avoid writing lcf data
        IF (config%output_diag) THEN
          CALL message(msg_info, "Writing additional diagnostic output \n")
          CALL ropp_pp_diag2roprof(diag, ro_data)
        ENDIF

        CALL ropp_io_addvar(ro_data,                               &   !HGL!
             name      = "LC_weight",                              &   !HGL!
             long_name = "SO-weight of observed LC bending angle", &   !HGL!
             units     = " ",                                      &   !HGL!
             range     = (/0.0_wp,1.0_wp/),                        &   !HGL!
             data      = WLC(imin:imax))                               !HGL!

       CALL message(msg_info, "Writing to output file " // TRIM(ofile) // "\n")
       CALL ropp_io_write(ro_data, ofile, append = .TRUE., ranchk = ranchk )

!-------------------------------------------------------------------------------
! 24. Clean up
!-------------------------------------------------------------------------------

        CALL ropp_io_free(ro_data)
        CALL ropp_io_free(smt_ba)
        CALL ropp_io_free(bangle)
        CALL ropp_io_free(out_ba)
        CALL ropp_io_free(out_refrac)
        CALL ropp_io_free(mod_ba)
        IF (ASSOCIATED(diag%ba_ion)) DEALLOCATE(diag%ba_ion)
        IF (ASSOCIATED(diag%err_ion)) DEALLOCATE(diag%err_ion)
        IF (ASSOCIATED(diag%err_neut)) DEALLOCATE(diag%err_neut)
        IF (ASSOCIATED(diag%CTimpact)) DEALLOCATE(diag%CTimpact)
        IF (ASSOCIATED(diag%CTamplitude)) DEALLOCATE(diag%CTamplitude)
        IF (ASSOCIATED(diag%CTamplitude_smt)) DEALLOCATE(diag%CTamplitude_smt)
        IF (ASSOCIATED(diag%CTimpactL2)) DEALLOCATE(diag%CTimpactL2)
        IF (ASSOCIATED(diag%CTamplitudeL2)) DEALLOCATE(diag%CTamplitudeL2)
        IF (ASSOCIATED(diag%CTamplitudeL2_smt)) DEALLOCATE(diag%CTamplitudeL2_smt)

        IF (ALLOCATED(WLC))    DEALLOCATE(WLC)       !HGL!
        
      END DO
    END DO    
  CONTAINS
    
!-------------------------------------------------------------------------------
! 25. Usage information
!-------------------------------------------------------------------------------
    
  SUBROUTINE usage()

    PRINT *, 'ropp_pp_occ_tool - Pre-processor tool'
    PRINT *, '                   Calculate corrected bending angle and '
    PRINT *, '                   refractivity from L1 and L2 phase data'
    PRINT *, 'ropp_pp_tool [<options>] <input_file(s)> -o <output_file>'
    PRINT *, 'Valid options are:'
    PRINT *, '  -h                give (this) help.'
    PRINT *, '  -o <output_file>  name of netCDF/ropp output file.'
    PRINT *, '  -c <config_file>  name of configuration file.'
    PRINT *, '  -m <method>       ionospheric correction method '
    PRINT *, '                    [NONE,MSIS,GMSIS,BG], (default MSIS).'
    PRINT *, '  -mfile <file>     model refractivity coefficients file path'
    PRINT *, '                    (default local search).'
    PRINT *, '  -bfile <file>     background model atmospheric profile file path'
    PRINT *, '                    (if using BG ionospheric correction method).'
    PRINT *, '  -navfile <file>   external navigation bit *_txt file path' 
    PRINT *, '                    (default internal correction).'
    PRINT *, '  -occ <occ_method> processing method, WO or GO (default WO)'
    PRINT *, '  -filter <filt_method> filtering method, slpoly or optest'
    PRINT *, '                         (default slpoly, sliding polynomial)'
    PRINT *, '  -fit             apply 2-parameter regression fit to model'
    PRINT *, '  -ellipsoid       output height with respect to WGS84 ellipsoid'
    PRINT *, '                         (default output wrt EGM96 geoid)'
    PRINT *, '  -d                 output additional diagnostics.'
    PRINT *, '  -no_ranchk         no range check performed before writing output'
    PRINT *, '  -V                give some version information.'
    PRINT *, ''

  END SUBROUTINE usage

!-------------------------------------------------------------------------------
! 26. Version information
!-------------------------------------------------------------------------------

  SUBROUTINE version_info(version)
    CHARACTER(len = *) :: version
    PRINT *, 'ropp_pp_occ_tool - Pre-processor occ tool: '
    PRINT *, '                   Calculate corrected bending angles '
    PRINT *, '                   and refractivity from excess phase'
    PRINT *, ''
    PRINT *, 'This program is part of ROPP version ' // TRIM(version)
    PRINT *, ''
  END SUBROUTINE version_info

END PROGRAM ropp_pp_occ_tool
