! $Id: ropp_fm_bg2ro_1d.f90 3020 2011-09-14 08:56:19Z frdo $

PROGRAM ropp_fm_bg2ro_1d

!****p* Programs/ropp_fm_bg2ro_1d *
!
! NAME
!   ropp_fm_bg2ro_1d
!
! SYNOPSIS
!   Calculate radio occultation pseudo observation from background model
!   data using 1d forward models
!
!   > ropp_fm_bg2ro_1d <infile(s)> -o <outfile> [-f] [-use_logp]
!                     [-use_logq] [-comp] [-d] [-h] [-v]
!
! ARGUMENTS
!   <infile(s)>   One (or more) input file names.
!
! OPTIONS
!   -o <outfile>  Name of output file (default: bg2ro.nc)
!   -f            forward only, no adjoint calculation.
!   -use_logp     use log(pressure) for forward model
!   -use_logq     use log(spec humidity) for forward model
!   -comp         include non ideal gas compressibility
!   -d            output additional diagnostics
!   -h            help
!   -v            version information
!
! DESCRIPTION
!   This program reads model data on model levels from the input data files
!   and calculates vertical profiles of bending angle and refractivity using
!   the 1d forward operators. The result is written to an ROPP formatted
!   output file.
!
! NOTES
!   If the input file is a multifile, or more than one input files are
!   specified, the output file is a multifile.
!
!   Already existing output files will be overwritten.
!
! EXAMPLE
!   To calculate bending angle and refractivity from one of the example
!   (single-) files in the data directory:
!
!     > ropp_fm_bg2ro_1d ../data/bgr20090401_000329_M02_2030337800_N0007_XXXX.nc
!
!   To calculate bending angle and refractivity profiles from all singlefiles
!   in the data directory:
!
!     > ropp_fm_bg2ro_1d ../data/bgr20090401*_N0007_XXXX.nc -o eg_02.nc
!
!   Note that the resulting eg_02.nc file contains forward modelled data from
!   all example profiles.
!
!   To calculate forward modelled bending angle and refractivity profiles from
!   all profiles contained in the multifile bgr20090401_multi.nc:
!
!     > ropp_fm_bg2ro_1d ../data/bgr20090401_multi.nc -o eg_03.nc
!
!   Since the ecmwf_multi_* file was generated by concatenating the other
!   files in the data directory, eg_02.nc and eg_03.nc should be identical
!   apart from the file names.
!
! SEE ALSO
!   ropp_fm_bangle_1d
!   ropp_fm_refrac_1d
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the GRAS SAF
!   Helpdesk at http://www.grassaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

!-------------------------------------------------------------------------------
! 1. Declarations
!-------------------------------------------------------------------------------

  USE typesizes, ONLY: wp => EightByteReal
  USE ropp_utils
  USE ropp_io
  USE ropp_io_types, ONLY: ROprof
  USE ropp_fm
  USE ropp_fm_types
  USE ropp_fm_copy

  IMPLICIT NONE

  TYPE(ROprof)                                     :: ro_data
  TYPE(State1dFM)                                  :: state
  TYPE(Obs1dRefrac)                                :: obs_refrac
  TYPE(Obs1dBangle)                                :: obs_bangle

  REAL(wp), DIMENSION(:,:), ALLOCATABLE            :: gradient_bangle
  REAL(wp), DIMENSION(:,:), ALLOCATABLE            :: gradient_refrac

  INTEGER                                          :: idummy
  INTEGER                                          :: i, iargc, argc, k
  INTEGER                                          :: n_files, n_profiles

  LOGICAL                                          :: give_help, do_adj
  LOGICAL                                          :: ranchk = .TRUE.
  LOGICAL                                          :: compress = .FALSE.

  CHARACTER(len = 4096), DIMENSION(:), ALLOCATABLE :: ifiles
  CHARACTER(len = 4096)                            :: ofile
  CHARACTER(len =  256)                            :: buffer
  CHARACTER(len =    4)                            :: istr
  CHARACTER(len =    6)                            :: nstr

!-------------------------------------------------------------------------------
! 2. Default settings
!-------------------------------------------------------------------------------

  do_adj    = .TRUE.
  give_help = .FALSE.
  ofile     = "bg2ro.nc"

  CALL message(msg_noin, '')
  CALL message(msg_noin, &
      '-----------------------------------------------------------------------')
  CALL message(msg_noin, &
      '                     ROPP Forward Model'                                )
  CALL message(msg_noin, &
      '-----------------------------------------------------------------------')
  CALL message(msg_noin, '')

!-------------------------------------------------------------------------------
! 3. Command line arguments
!-------------------------------------------------------------------------------

  argc = iargc()
  i = 1
  n_files = 0
  ALLOCATE(ifiles(argc))

  DO WHILE(i <= argc)
    CALL getarg(i, buffer)
    SELECT CASE (buffer)
        CASE('-o')                          ! Output file name (netCDF output)
           CALL getarg(i+1, buffer)
           ofile = buffer
           i = i + 1
        CASE ('-f')                         ! Perform forward model only
           do_adj = .FALSE.
        CASE ('-comp')
           compress = .TRUE.                ! non ideal gas switch
        CASE ('-use_logp')                  ! Use log(pressure) for FM
           state%use_logp = .TRUE.
        CASE ('-use_logq')                  ! Use log(shum) for FM
           state%use_logq = .TRUE.
        CASE ('-no-ranchk')                 ! Use no rangecheck on output
           ranchk = .FALSE.
        CASE('-d')                          ! Additional diagnostic mode
           msg_MODE = VerboseMode
        CASE('-h', '--help', '?')           ! help
           give_help = .TRUE.
        CASE('-v', '-V', '--version')       ! version information
           CALL version_info()
           CALL EXIT(0)
        CASE default                        ! Input file name
           IF ( buffer(1:1) /= '-' ) THEN
             n_files = n_files + 1
             ifiles(n_files) = buffer
           END IF
    END SELECT
    i = i + 1
  END DO

  IF (argc == 0 .OR. n_files == 0 .OR. give_help) THEN
    CALL usage()
    CALL EXIT(0)
  ENDIF

!-------------------------------------------------------------------------------
! 4. Remove pre-existing output file
!-------------------------------------------------------------------------------

  CALL file_delete(ofile, idummy)

!-------------------------------------------------------------------------------
! 4. Set default units
!-------------------------------------------------------------------------------

  CALL ropp_fm_set_units(ro_data)

!-------------------------------------------------------------------------------
! 5. Loop over all input files
!-------------------------------------------------------------------------------

  DO k = 1, n_files

!-------------------------------------------------------------------------------
! 6. Loop over all profiles
!-------------------------------------------------------------------------------

    n_profiles = ropp_io_nrec(ifiles(k))

    DO i = 1, n_profiles

      WRITE(istr, '(i4)') i
      WRITE(nstr, '(i6)') n_profiles
      CALL message(msg_noin, '')
      CALL message(msg_info, "Processing profile " // istr // " of " // nstr )

!-------------------------------------------------------------------------------
! 7. Read data
!-------------------------------------------------------------------------------

      CALL ropp_io_read(ro_data, ifiles(k), rec = i, ranchk = ranchk)
      CALL message(msg_info, "(" // TRIM(ro_data%occ_id) // ") \n")

!-------------------------------------------------------------------------------
! 8. Copy data in RO structure to state and refrac obs vectors
!-------------------------------------------------------------------------------

      CALL ropp_fm_roprof2state(ro_data, state)

! switch non-ideal gas on

      IF (compress) state%non_ideal = .TRUE.

      IF (ro_data%lev2a%Npoints > 0) THEN
        CALL ropp_fm_roprof2obs(ro_data, obs_refrac)    !!set to obs levels
        obs_refrac%refrac(:) = ropp_MDFV
      ELSE
        CALL set_obs_levels_refrac(ro_data, obs_refrac) !!pre-defined levels
      ENDIF

!-------------------------------------------------------------------------------
! 9. Calculate refractivity and its gradient
!-------------------------------------------------------------------------------

      IF (state%state_ok) THEN

        CALL ropp_fm_refrac_1d(state, obs_refrac)

        IF (do_adj) THEN
          ALLOCATE(gradient_refrac(SIZE(obs_refrac%refrac),SIZE(state%state)))
          CALL ropp_fm_refrac_1d_grad(state, obs_refrac, gradient_refrac)
        ENDIF

      ENDIF

!-------------------------------------------------------------------------------
! 10. Copy data in RO and refrac structure to bending angle obs vector
!-------------------------------------------------------------------------------

      IF (ro_data%lev1b%Npoints > 0) THEN
        CALL ropp_fm_roprof2obs(ro_data, obs_bangle)   !!set to obs levels
        obs_bangle%bangle(:) = ropp_MDFV
      ELSE
        CALL set_obs_levels_bangle(ro_data, obs_refrac, obs_bangle) !default
      ENDIF

!-------------------------------------------------------------------------------
! 11. Calculate bending angle and its gradient
!-------------------------------------------------------------------------------

      IF (state%state_ok) THEN

        CALL ropp_fm_bangle_1d(state, obs_bangle)
        IF (do_adj) THEN
          ALLOCATE(gradient_bangle(SIZE(obs_bangle%bangle),SIZE(state%state)))
          CALL ropp_fm_bangle_1d_grad(state, obs_bangle, gradient_bangle)
        ENDIF

      ENDIF

!-------------------------------------------------------------------------------
! 12. Copy simulated observations to RO structure and write data
!-------------------------------------------------------------------------------

      CALL ropp_fm_obs2roprof(obs_refrac, ro_data)
      CALL ropp_fm_obs2roprof(obs_bangle, ro_data)

      IF (state%state_ok) THEN

      IF (do_adj) THEN
        CALL ropp_io_addvar(ro_data,                                      &
                            name      = "gradient_refrac",                &
                            long_name =                                   &
                            "Gradient of the refractivity forward model", &
                            units     = "1",                              &
                            range     = (/MINVAL(gradient_refrac),        &
                                          MAXVAL(gradient_refrac)/),      &
                            DATA      = gradient_refrac)

        CALL ropp_io_addvar(ro_data,                                       &
                            name      = "gradient_bangle" ,                &
                            long_name =                                    &
                            "Gradient of the bending angle forward model", &
                            units     = "rad",                             &
                            range     = (/MINVAL(gradient_bangle),         &
                                          MAXVAL(gradient_bangle)/),       &
                            DATA      = gradient_bangle)
      ENDIF

!-------------------------------------------------------------------------------
! 13. Update RO structure with computed state vector variables
!-------------------------------------------------------------------------------

      CALL ropp_fm_state2roprof(state, ro_data)

      ENDIF

!-------------------------------------------------------------------------------
! 14. Write data
!-------------------------------------------------------------------------------

      CALL ropp_io_write(ro_data, ofile, append = .TRUE., ranchk = .TRUE. )

!-------------------------------------------------------------------------------
! 15. Clean up
!-------------------------------------------------------------------------------

      IF (state%state_ok .AND. do_adj) THEN
        DEALLOCATE (gradient_refrac)
        DEALLOCATE (gradient_bangle)
      ENDIF

      CALL ropp_fm_free(state)
      CALL ropp_fm_free(obs_refrac)
      CALL ropp_fm_free(obs_bangle)
      CALL ropp_io_free(ro_data)

    END DO
  END DO

CONTAINS

!-------------------------------------------------------------------------------
! 16. Calculate observation levels for refractivity
!-------------------------------------------------------------------------------

  SUBROUTINE set_obs_levels_refrac(ro_data, obs_refrac)

!   x.1 Declarations
!   ----------------

    USE typesizes, ONLY: wp => EightByteReal
    USE ropp_io_types
    USE ropp_fm

    IMPLICIT NONE

    TYPE(ROprof)      :: ro_data
    TYPE(Obs1dRefrac) :: obs_refrac

    INTEGER           :: i, n, dummy

    dummy = ro_data%Lev2a%Npoints !! fix nag ro_data 'unused dummy variable'

!   x.2 Vertical geopotential height levels between 200 and 60000 gpm
!   -----------------------------------------------------------------

    n = INT(60.0_wp / 0.2_wp)

    ALLOCATE(obs_refrac%refrac(n))
    ALLOCATE(obs_refrac%geop(n))
    ALLOCATE(obs_refrac%weights(n))

    obs_refrac%refrac(:)  = 0.0_wp
    obs_refrac%geop(:)    = (/ (i*200.0_wp, i = 1,n) /)
    obs_refrac%weights(:) = 1.0_wp

  END SUBROUTINE set_obs_levels_refrac


!-------------------------------------------------------------------------------
! 17. Calculate obs levels for bending angle (consistent with obs_refrac)
!-------------------------------------------------------------------------------

  SUBROUTINE set_obs_levels_bangle(ro_data, obs_refrac, obs_bangle)

!   x.1 Declarations
!   ----------------

    USE typesizes, ONLY: wp => EightByteReal
    USE geodesy
    USE ropp_io_types
    USE ropp_fm

    IMPLICIT NONE

    TYPE(ROprof)      :: ro_data
    TYPE(Obs1dRefrac) :: obs_refrac
    TYPE(Obs1dbangle) :: obs_bangle

    REAL(wp), DIMENSION(:), ALLOCATABLE :: tmp

    INTEGER           :: n

!   x.2 Allocate arrays
!   -------------------

    n = SIZE(obs_refrac%geop)

    ALLOCATE(obs_bangle%bangle(n))
    ALLOCATE(obs_bangle%impact(n))
    ALLOCATE(obs_bangle%weights(n))

    ALLOCATE(tmp(n))

!   x.3 Set scalar arguments of the observation vector
!   --------------------------------------------------

    obs_bangle%g_sfc        = gravity(ro_data%GEOref%lat)
    obs_bangle%r_earth      = R_eff(ro_data%GEOref%lat)
    obs_bangle%undulation   = ro_data%GEOref%undulation
    IF (ro_data%GEOref%roc > 0.0_wp) THEN
      obs_bangle%r_curve = ro_data%GEOref%roc
    ELSE
      obs_bangle%r_curve = obs_bangle%r_earth
    ENDIF

!   x.4 Calculate levels to coincide with the geopotential levels
!   -------------------------------------------------------------

    tmp = geopotential2geometric(ro_data%GEOref%lat, obs_refrac%geop) &
            + obs_bangle%r_curve + obs_bangle%undulation

    obs_bangle%impact(:)  = (1.0_wp + obs_refrac%refrac*1.e-6_wp) * tmp

!   x.5 Fill other arrays
!   ---------------------

    obs_bangle%bangle(:)  = 0.0_wp
    obs_bangle%weights(:) = 1.0_wp

!   x.6 Clean up
!   ------------

    DEALLOCATE(tmp)

  END SUBROUTINE set_obs_levels_bangle

!-------------------------------------------------------------------------------
! 18. Usage information
!-------------------------------------------------------------------------------

  SUBROUTINE usage()
    PRINT *, 'Purpose:'
    PRINT *, '  Bending angles and refractivity forward model'
    PRINT *, 'Usage:'
    PRINT *, '  > ropp_fm_bg2ro [<options>] <input_file(s)>'
    PRINT *, 'Options:'
    PRINT *, '  -o <output_file> name of ROPP netCDF output file'
    PRINT *, '  -f               forward only, no adjoint calculation.'
    PRINT *, '  -use_logp        use log(pressure) for forward model'
    PRINT *, '  -use_logq        use log(spec humidity) for forward model'
    PRINT *, '  -comp            include non ideal gas compressibility'
    PRINT *, '  -d               output additional diagnostics'
    PRINT *, '  -h               this help'
    PRINT *, '  -v               version information'
    PRINT *, ''
  END SUBROUTINE usage

!-------------------------------------------------------------------------------
! 19. Version information
!-------------------------------------------------------------------------------

  SUBROUTINE version_info()
    CHARACTER (LEN=40) :: version
    version = ropp_fm_version()
    PRINT *, 'ropp_fm_bg2ro_1d - Bending angles and refractivity forward model.'
    PRINT *, ''
    PRINT *, 'This program is part of ROPP (FM) Release ' // TRIM(version)
    PRINT *, ''
  END SUBROUTINE version_info

END PROGRAM ropp_fm_bg2ro_1d
