PROGRAM robufr2ropp

!****pi* Tests/robufr2ropp *

! NAME
!    robufr2ropp - Reads BUFR file holding single profile of
!    RO data and converts to ROPP netCDF format.
!
! SYNOPSIS
!    robufr2ropp  infile.bfr  -o ofile.nc [-d]
!
! DESCRIPTION
!    Emulates bufr2ropp but avoids the use of BUFR libraries.
!    This allows us to test the results of ropp2bufr without incurring
!    the circular logic of depending on bufr2ropp, which could have
!    undergone the same changes as ropp2bufr.  It does this by chopping up
!    the bitstream that constitutes Sec. 4 of a BUFR message into chunks,
!    each one representing a particular datum.  The size of the chunks is
!    taken from the ROM SAF BUFR document.
!
! INPUT
!    Single profile RO BUFR message.  ARH and FTP headers are allowed.
!
! OUTPUT
!    Single profile ROPP format netCDF file.  If the '-d' option is used,
!    a detailed bit dump of the input message is written to stdout.  This
!    allows very close checking of the results of the bufr2ropp, which
!    can be built with a variety of BUFR libraries.
!
! NOTES
!    Could have application as a debugging too in its own right.
!    Probably slower than bufr2ropp.
!
! REFERENCES
!   WMO FM94 (BUFR) Specification For Radio Occultation Data,
!   SAF/ROM/METO/FMT/BUFR/001
!
! AUTHOR
!    Met Office, Exeter, UK.
!    Any comments on this software should be given via the ROM SAF
!    Helpdesk at http://www.romsaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

  USE messages
  USE ropp_utils
  USE ropp_io
  USE ropp_io_types
  USE geodesy,       ONLY: geometric2geopotential

  IMPLICIT NONE

  TYPE(ROprof)             :: ro_data

  TYPE sec4_type
    CHARACTER(LEN=32)      :: name=''
    INTEGER                :: nbits=31
    INTEGER                :: shift=0
    INTEGER                :: scale=0
  END TYPE sec4_type

  CHARACTER(LEN=256)       :: ifile='robufr2ropp_in.bfr'  ! (Default) input file name
  CHARACTER(LEN=256)       :: ofile='robufr2ropp_out.nc'  ! (Default) output file name
  CHARACTER(LEN=256)       :: arg                         ! Command line arguments

  LOGICAL                  :: padding = .FALSE.

  INTEGER                  :: iargc, iarg, narg

!-------------------------------------------------------------
! 1. Initialise
!-------------------------------------------------------------

  CALL message_set_routine ( 'robufr_read' )

  CALL message(msg_noin, '')
  CALL message(msg_noin, &
       '---------------------------------------------------------------------')
  CALL message(msg_noin, &
       '                       ROBUFR to ROPP converter'                      )
  CALL message(msg_noin, &
       '---------------------------------------------------------------------')
  CALL message(msg_noin, '')

!-------------------------------------------------------------
! 2. Parse command line options
!-------------------------------------------------------------

  narg = IARGC()

  ifile      = 'robufr2ropp_in.bfr'  ! Default input file name
  ofile      = 'robufr2ropp_out.nc'  ! Default output file name

  iarg = 1

  DO WHILE ( iarg <= narg )

    CALL GETARG ( iarg, arg )

    SELECT CASE (arg)

      CASE ('-d', '-D', '--debug')
        msg_MODE = VerboseMode

      CASE ('-o', '-O', '--output')
        iarg = iarg + 1
        CALL GETARG ( iarg, arg )
        ofile = arg

      CASE DEFAULT
         IF ( arg(1:1) /= '-' ) THEN
           ifile = arg
         END IF

    END SELECT

    iarg = iarg + 1

  END DO

  IF ( ifile == ' ' ) THEN
    CALL message ( msg_error, 'No input file(s) specified' )
    narg = 0
  END IF

  IF ( narg == 0 ) THEN
    CALL EXIT(msg_exit_status)
  ENDIF

  CALL message(msg_noin, '')
  CALL message(msg_noin, 'Converting ' // TRIM(ADJUSTL(ifile)) // &
                         ' to ' // TRIM(ADJUSTL(ofile)))

! 3.0 Read input file into ROprof structure
! -----------------------------------------

  CALL robufr_read(ifile, ro_data)

! 4.0 Write ROprof structure into netCDF file
! -------------------------------------------

  CALL ropp_io_write(ro_data, ofile)

! 5.0 Clean up
! ------------

  CALL ropp_io_free(ro_data)


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE robufr_read(file, ro_data)

    CHARACTER(LEN=256), INTENT(in)           :: file
    TYPE(ROprof), INTENT(inout)              :: ro_data

    INTEGER, PARAMETER                       :: n_bytes_max=50000         ! Max size of bufr file in bytes
    INTEGER, PARAMETER                       :: n_sec4_fields_max=50000   ! Max number of data values
    CHARACTER(LEN=n_bytes_max)               :: cdata

    INTEGER                                  :: file_id=11  ! Should never need this default
    INTEGER, PARAMETER                       :: dp=KIND(1.D0)
    INTEGER, PARAMETER                       :: ropp_mifv=-999
    REAL(dp), PARAMETER                      :: ropp_mdfv=-9.9999E7_dp
    INTEGER                                  :: n_lev1b, n_lev2a, n_lev2b, n_lev2c, n_freq
    INTEGER                                  :: n_sec4_fields

    INTEGER                                  :: n_bytes     ! size of bufr file in bytes
    INTEGER                                  :: n_edn       ! BUFR edition
    INTEGER                                  :: wrapper_length=0  ! no. of bytes in ARH(+FTP) wrapper
    INTEGER                                  :: i, j, ifirst, ilast

    INTEGER                                  :: ivalue
    REAL(dp)                                 :: rvalue

    INTEGER                                  :: idata(n_sec4_fields_max)
    REAL(dp)                                 :: rdata(n_sec4_fields_max)

    CHARACTER(LEN=128)                       :: msg

    TYPE(sec4_type), ALLOCATABLE             :: sec4_data(:)

    CHARACTER(LEN=8)                         :: ctemp
    CHARACTER(LEN=8*n_bytes_max)             :: sec4_bitstream
    INTEGER                                  :: n_sec4_octets
    CHARACTER(LEN=13)                        :: s_octet

!-------------------------------------------------------------
! 1. Initialise
!-------------------------------------------------------------

    idata(:) = ropp_mifv
    rdata(:) = ropp_mdfv

! 2.0 Check whether we need to strip off the ARH and the FTP header
! -----------------------------------------------------------------

    file_id = get_io_unit()
    OPEN (file_id, FILE=file, STATUS='OLD', FORM='UNFORMATTED', ACCESS='DIRECT', RECL=4)
    READ (file_id, rec=1) cdata(1:4)
    CLOSE (file_id)

    SELECT CASE ( cdata(1:4) )
      CASE ( 'BUFR' ) ! 'pure' BUFR message
        wrapper_length = 0
      CASE ( char(1)//char(13)//char(13)//char(10) ) ! BUFR message wrapped in ARH
        wrapper_length = 31
      CASE DEFAULT ! BUFR message wrapped in ARH+FTP, we assume - this could be improved.
        wrapper_length = 41
    END SELECT

! 3.0 Extract message length and BUFR edition from bytes 5-7 and byte 8 of bufr message
! -------------------------------------------------------------------------------------

    file_id = get_io_unit()
    OPEN (file_id, FILE=file, STATUS='OLD', FORM='UNFORMATTED', ACCESS='DIRECT', RECL=8+wrapper_length)
    READ (file_id, rec=1) cdata(1:8+wrapper_length)
    CLOSE (file_id)

    CALL BUFR_decode_sample(cdata(1+wrapper_length:8+wrapper_length), 'Message length', 5, 7, size=n_bytes, quiet=.TRUE.)
    CALL BUFR_decode_sample(cdata(1+wrapper_length:8+wrapper_length), 'BUFR edition', 8, 8, size=n_edn, quiet=.TRUE.)
    IF ( n_edn /= 4 ) THEN
      WRITE(msg, '(A,I1,A)') 'BUFR Ed. = ', n_edn, ' ... can only process BUFR Ed. 4 data currently.'
      CALL message(msg_fatal, ADJUSTL(msg))
    ENDIF

! 4.0 Read entire dataset as sequence of 1-byte characters
! --------------------------------------------------------

    file_id = get_io_unit()
    OPEN (file_id, FILE=file, STATUS='OLD', FORM='UNFORMATTED', ACCESS='DIRECT', RECL=n_bytes+wrapper_length)
    READ (file_id, rec=1) cdata(1:n_bytes+wrapper_length)
    CLOSE (file_id)

! 5.0 IF ARH or FTP present, strip them off cdata
! -----------------------------------------------

    IF ( wrapper_length > 0 ) cdata(1:n_bytes) = cdata(1+wrapper_length:n_bytes+wrapper_length)

! 6.0 Write out contents of file
! ------------------------------

    IF ( msg_MODE == VerboseMode ) THEN

      CALL message(msg_noin, '')
      CALL message(msg_noin, '-------------------------------------------------------------------')
      CALL message(msg_noin, ' Contents of ' // TRIM(ADJUSTL(file)))
      CALL message(msg_noin, '-------------------------------------------------------------------')
      CALL message(msg_noin, '')

! 6.1 Section 0
      CALL message(msg_noin, '')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, ' Contents of Sec 0 ')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, '  octet                       name   value  ')
      CALL message(msg_noin, '--------------------------------------------')

      CALL message(msg_noin, '   1- 4                       BUFR    ' // cdata(1:4))
      CALL BUFR_decode_sample(cdata, 'Message length',           5,  7)
      CALL BUFR_decode_sample(cdata, 'BUFR edition',             8,  8)
      CALL message(msg_noin, '')

! 6.2 Section 1 (Ed. 4)
      CALL message(msg_noin, '')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, ' Contents of Sec 1 ')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, '  octet                       name   value  ')
      CALL message(msg_noin, '--------------------------------------------')

      CALL BUFR_decode_sample(cdata, 'Length of section 1',      9, 11)
      CALL BUFR_decode_sample(cdata, 'BUFR Master Table',       12, 12)
      CALL BUFR_decode_sample(cdata, 'Originating Centre',      13, 14)
      CALL BUFR_decode_sample(cdata, 'Originating Subcentre',   15, 16)
      CALL BUFR_decode_sample(cdata, 'Update sequence Number',  17, 17)
      CALL BUFR_decode_sample(cdata, 'Optional section 2 flag', 18, 18)
      CALL BUFR_decode_sample(cdata, 'Data category (Table A)', 19, 19)
      CALL BUFR_decode_sample(cdata, 'Intnl data sub-category', 20, 20)
      CALL BUFR_decode_sample(cdata, 'Local data sub-category', 21, 21)
      CALL BUFR_decode_sample(cdata, 'Version of Master Table', 22, 22)
      CALL BUFR_decode_sample(cdata, 'Version of Local Table',  23, 23)
      CALL BUFR_decode_sample(cdata, 'Year',                    24, 25)
      CALL BUFR_decode_sample(cdata, 'Month',                   26, 26)
      CALL BUFR_decode_sample(cdata, 'Day',                     27, 27)
      CALL BUFR_decode_sample(cdata, 'Hour',                    28, 28)
      CALL BUFR_decode_sample(cdata, 'Minute',                  29, 29)
      CALL BUFR_decode_sample(cdata, 'Second',                  30, 30)
      CALL message(msg_noin, '')

! 6.3 Section 3
      CALL message(msg_noin, '')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, ' Contents of Sec 3 ')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, '  octet                       name   value  ')
      CALL message(msg_noin, '--------------------------------------------')
      CALL BUFR_decode_sample(cdata, 'Length of section 3',     31, 33)
      CALL BUFR_decode_sample(cdata, 'Reserved',                34, 34)
      CALL BUFR_decode_sample(cdata, 'Number of datasets',      35, 36)
      CALL BUFR_decode_sample(cdata, 'Section 4 data flags',    37, 37)
      CALL BUFR_decode_sample(cdata, 'Descriptor',              38, 39)
      CALL BUFR_decode_sample(cdata, 'Pad byte',                40, 40)
      CALL message(msg_noin, '')

    ENDIF !  IF ( msg_MODE == VerboseMode )

    CALL BUFR_decode_sample(cdata, 'Length of section 3',     31, 33, pad=padding, quiet=.TRUE.)
    IF ( padding ) cdata(40:n_bytes-1) = cdata(41:n_bytes)  ! remove padding octet

! 6.4 Section 4
    IF ( msg_MODE == VerboseMode ) THEN

      CALL message(msg_noin, '')
      CALL message(msg_noin, '----------------------------')
      CALL message(msg_noin, ' Contents of Sec 4 (header) ')
      CALL message(msg_noin, '----------------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, '  octet                       name   value  ')
      CALL message(msg_noin, '--------------------------------------------')
      CALL BUFR_decode_sample(cdata, 'Length of section 4',     40, 42)
      CALL BUFR_decode_sample(cdata, 'Reserved octet of sec 4',   43, 43)
      CALL message(msg_noin, '')

      CALL message(msg_noin, '')
      CALL message(msg_noin, '--------------------------')
      CALL message(msg_noin, ' Contents of Sec 4 (data) ')
      CALL message(msg_noin, '--------------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, &
        '  field  name               shift      ref. val                      binary value      real value  ')
      CALL message(msg_noin, &
        '---------------------------------------------------------------------------------------------------')

    ENDIF !  IF ( msg_MODE == VerboseMode )

    CALL BUFR_decode_sample(cdata, 'Length of section 4',     40, 42, size=n_sec4_octets, quiet=.TRUE.)

! 6.4.1 Read ALL the data in Sec 4 into one long bitstream, sec4_bitstream
    DO i=1, n_sec4_octets
      ctemp = '00000000'
      DO j=0,7
        IF ( BTEST(ICHAR(cdata(i+43:i+43)), j) ) ctemp(8-j:8-j) = '1'
      ENDDO
      sec4_bitstream(8*i-7:8*i) = ctemp
    ENDDO

! 6.4.2 Extract dimensions from sec4_bitstream
!       All ifirst and ilasts calculated from field widths in ROM SAF BUFR doc
    ifirst = 824 + 1  ; ilast = ifirst + 8 - 1
    n_freq = ival(sec4_bitstream(ifirst:ilast))

    ifirst = 741 + 1  ; ilast = ifirst + 16 - 1
    n_lev1b = ival(sec4_bitstream(ifirst:ilast))

    ifirst = 757 + (82 + 84*n_freq)*n_lev1b + 1  ; ilast = ifirst + 16 - 1
    n_lev2a = ival(sec4_bitstream(ifirst:ilast))

    ifirst = 773 + (82 + 84*n_freq)*n_lev1b  + 69*n_lev2a + 1  ; ilast = ifirst + 16 - 1
    n_lev2b = ival(sec4_bitstream(ifirst:ilast))

    n_lev2c = 0

    n_sec4_fields = 47 + (5 + 6*n_freq)*n_lev1b + 6*n_lev2a + 10*n_lev2b

! 6.4.3 Initialise sec4_data and metadata
    ALLOCATE (sec4_data(n_sec4_fields))
    CALL BUFR_sec4_init(sec4_data, n_lev1b, n_lev2a, n_lev2b, n_freq)

! 6.4.4 Populate idata/rdata with the decoded fields
    ifirst = 0  ;  ilast = 0
    DO i=1, n_sec4_fields
      ifirst = ilast + 1  ;  ilast = ifirst + sec4_data(i)%nbits - 1
      IF ( sec4_data(i)%scale == 0 ) THEN  ! integer
        IF ( sec4_bitstream(ifirst:ilast) == REPEAT('1', ilast-ifirst+1) ) THEN
          ivalue = ropp_mifv
        ELSE
          ivalue = ival(sec4_bitstream(ifirst:ilast)) + sec4_data(i)%shift
        ENDIF
        idata(i) = ivalue
        IF ( msg_MODE == VerboseMode ) THEN
          WRITE(msg, '(I6,2X,A20,2X,I2,2X,I12,2X,A32,I16)') &
            i, sec4_data(i)%name, sec4_data(i)%scale, sec4_data(i)%shift, sec4_bitstream(ifirst:ilast), ivalue
          CALL message(msg_noin, ' ' // msg)
        ENDIF
      ELSE  ! real(dp)
        IF ( sec4_bitstream(ifirst:ilast) == REPEAT('1', ilast-ifirst+1) ) THEN
          rvalue = ropp_mdfv
        ELSE
          rvalue = (ival(sec4_bitstream(ifirst:ilast)) + sec4_data(i)%shift) / (10.0_dp**sec4_data(i)%scale)
        ENDIF
        rdata(i) = rvalue
        IF ( msg_MODE == VerboseMode ) THEN
          WRITE(msg, '(I6,2X,A20,2X,I2,2X,I12,2X,A32,G20.10)') &
            i, sec4_data(i)%name, sec4_data(i)%scale, sec4_data(i)%shift, sec4_bitstream(ifirst:ilast), rvalue
          CALL message(msg_noin, ' ' // msg)
        ENDIF
      ENDIF
    ENDDO
    CALL message(msg_noin, '')

! 6.4.5 Initialise RO profile
    CALL ropp_io_free(ro_data)
    CALL ropp_io_init(ro_data, 0, n_lev1b, n_lev2a, n_lev2b, n_lev2c, 0)

! 6.4.6 Fill RO profile with Sec 4 data
    CALL BUFR_sec4_to_ROPP(ro_data, idata, rdata)

! 6.5 Section 5
! (This won't be affected by the possible removing of the padding byte in Sec 6.3.)
    IF ( msg_MODE == VerboseMode ) THEN

      CALL message(msg_noin, '')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, ' Contents of Sec 5 ')
      CALL message(msg_noin, '-------------------')
      CALL message(msg_noin, '')
      CALL message(msg_noin, '  octet                       name   value  ')
      CALL message(msg_noin, '--------------------------------------------')
      WRITE(s_octet, '(I6,A1,I6)') n_bytes-3, '-', n_bytes
      WRITE(msg, '(A)') s_octet // '                7777    ' // cdata(n_bytes-3:n_bytes)
      CALL message(msg_noin, '  ' // ADJUSTL(msg))
      CALL message(msg_noin, '')

    ENDIF  !  IF ( msg_MODE == VerboseMode )

    DEALLOCATE (sec4_data)

! 6.6 Write summary of input file
    CALL message(msg_noin, 'Summary of input BUFR file:')

    SELECT CASE (wrapper_length)
      CASE (  0 )
        CALL message(msg_cont, 'Contains a BUFR message without an ARH or FTP header')
      CASE ( 31 )
        CALL message(msg_cont, 'Contains a BUFR message wrapped in an ARH')
      CASE ( 41 )
        CALL message(msg_cont, 'Contains a BUFR message wrapped in an ARH and an FTP header')
    END SELECT

    WRITE(msg, '(A,I6,A)') 'Message length = ', n_bytes, ' bytes'
    CALL message(msg_cont, ADJUSTL(msg))

    WRITE(msg, '(A,I1)') 'BUFR edition = ', n_edn
    CALL message(msg_cont, ADJUSTL(msg))

    WRITE(msg, '(A,I1)') 'No. of frequencies = ', n_freq
    CALL message(msg_cont, ADJUSTL(msg))

    WRITE(msg, '(A,I4)') 'No. of level 1b variables = ', n_lev1b
    CALL message(msg_cont, ADJUSTL(msg))

    WRITE(msg, '(A,I4)') 'No. of level 2a variables = ', n_lev2a
    CALL message(msg_cont, ADJUSTL(msg))

    WRITE(msg, '(A,I4)') 'No. of level 2b variables = ', n_lev2b
    CALL message(msg_cont, ADJUSTL(msg))

  END SUBROUTINE robufr_read


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE BUFR_sec4_init(sec4_data, n_lev1b, n_lev2a, n_lev2b, n_freq)
!
! Initialise the sec4_data structure, which holds the metadata for each Sec 4 field.
!
    TYPE(sec4_type), INTENT(inout)     :: sec4_data(:)
    INTEGER, INTENT(in)                :: n_lev1b, n_lev2a, n_lev2b, n_freq

    INTEGER                            :: index, i, j

! Headers
! -------

    index = 0

    index = index + 1
    sec4_data(index)%name  = 'LEO ID'
    sec4_data(index)%nbits = 10
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Instrument ID'
    sec4_data(index)%nbits = 11
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Orig centre'
    sec4_data(index)%nbits =  8
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Product type'
    sec4_data(index)%nbits =  8
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Software ID'
    sec4_data(index)%nbits = 14
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Time significance'
    sec4_data(index)%nbits =  5
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Year'
    sec4_data(index)%nbits = 12
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Month'
    sec4_data(index)%nbits =  4
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Day'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Hour'
    sec4_data(index)%nbits =  5
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Minute'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Second'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  3
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Quality flags'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Percent confidence'
    sec4_data(index)%nbits =  7
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'LEO x coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'LEO y coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'LEO z coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'LEO x vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'LEO y vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'LEO z vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS ID'
    sec4_data(index)%nbits =  9
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'GNSS PRN'
    sec4_data(index)%nbits = 17
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'GNSS x coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  1
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS y coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  1
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS z coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  1
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS x vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS y vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'GNSS z vel'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'Time inc'
    sec4_data(index)%nbits = 18
    sec4_data(index)%scale =  3
    sec4_data(index)%shift = -4096

    index = index + 1
    sec4_data(index)%name  = 'Latitude'
    sec4_data(index)%nbits = 25
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -9000000

    index = index + 1
    sec4_data(index)%name  = 'Longitude'
    sec4_data(index)%nbits = 26
    sec4_data(index)%scale =  5
    sec4_data(index)%shift = -18000000

    index = index + 1
    sec4_data(index)%name  = 'CoC x coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'CoC y coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'CoC z coord'
    sec4_data(index)%nbits = 31
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -1073741824

    index = index + 1
    sec4_data(index)%name  = 'RoC'
    sec4_data(index)%nbits = 22
    sec4_data(index)%scale =  1
    sec4_data(index)%shift = 62000000

    index = index + 1
    sec4_data(index)%name  = 'Azimuth'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  2
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Undulation'
    sec4_data(index)%nbits = 15
    sec4_data(index)%scale =  2
    sec4_data(index)%shift = -15000

! Lev1b
! -----

    index = index + 1
    sec4_data(index)%name  = 'No. of level 1b data'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    IF ( n_lev1b > 0 ) THEN

      DO i=1,n_lev1b

        index = index + 1
        sec4_data(index)%name  = 'Latitude_tp'
        sec4_data(index)%nbits = 25
        sec4_data(index)%scale =  5
        sec4_data(index)%shift = -9000000

        index = index + 1
        sec4_data(index)%name  = 'Longitude_tp'
        sec4_data(index)%nbits = 26
        sec4_data(index)%scale =  5
        sec4_data(index)%shift = -18000000

        index = index + 1
        sec4_data(index)%name  = 'Azimuth_tp'
        sec4_data(index)%nbits = 16
        sec4_data(index)%scale =  2
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'No. of freqs'
        sec4_data(index)%nbits =  8
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

        DO j=1,n_freq

            index = index + 1
            sec4_data(index)%name  = 'Nominal freq'
            sec4_data(index)%nbits =  7
            sec4_data(index)%scale = -8
            sec4_data(index)%shift =  0

            index = index + 1
            sec4_data(index)%name  = 'Impact param'
            sec4_data(index)%nbits = 22
            sec4_data(index)%scale =  1
            sec4_data(index)%shift = 62000000

            index = index + 1
            sec4_data(index)%name  = 'Bending angle'
            sec4_data(index)%nbits = 23
            sec4_data(index)%scale =  8
            sec4_data(index)%shift = -100000

            index = index + 1
            sec4_data(index)%name  = 'First order stats'
            sec4_data(index)%nbits =  6
            sec4_data(index)%scale =  0
            sec4_data(index)%shift =  0

            index = index + 1
            sec4_data(index)%name  = 'Bangle error'
            sec4_data(index)%nbits = 20
            sec4_data(index)%scale =  8
            sec4_data(index)%shift = -100000

            index = index + 1
            sec4_data(index)%name  = 'First order stats'
            sec4_data(index)%nbits =  6
            sec4_data(index)%scale =  0
            sec4_data(index)%shift =  0

        ENDDO

        index = index + 1
        sec4_data(index)%name  = 'Percent confidence'
        sec4_data(index)%nbits =  7
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

      ENDDO

    ENDIF

! Lev2a
! -----

    index = index + 1
    sec4_data(index)%name  = 'No. of level 2a data'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    IF ( n_lev2a > 0 ) THEN

      DO i=1,n_lev2a

        index = index + 1
        sec4_data(index)%name  = 'Geom altitude'
        sec4_data(index)%nbits = 17
        sec4_data(index)%scale =  0
        sec4_data(index)%shift = -1000

        index = index + 1
        sec4_data(index)%name  = 'Refrac'
        sec4_data(index)%nbits = 19
        sec4_data(index)%scale =  3
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'First order stats'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Refrac error'
        sec4_data(index)%nbits = 14
        sec4_data(index)%scale =  3
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'First order stats'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Percent confidence'
        sec4_data(index)%nbits =  7
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

      ENDDO

    ENDIF

! Lev2b
! -----

    index = index + 1
    sec4_data(index)%name  = 'No. of level 2b data'
    sec4_data(index)%nbits = 16
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    IF ( n_lev2b >  0 ) THEN

      DO i=1,n_lev2b

        index = index + 1
        sec4_data(index)%name  = 'Geop height'
        sec4_data(index)%nbits = 17
        sec4_data(index)%scale =  0
        sec4_data(index)%shift = -1000

        index = index + 1
        sec4_data(index)%name  = 'Pressure'
        sec4_data(index)%nbits = 14
        sec4_data(index)%scale = -1
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Temperature'
        sec4_data(index)%nbits = 12
        sec4_data(index)%scale =  1
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Spec hum'
        sec4_data(index)%nbits = 14
        sec4_data(index)%scale =  5
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'First order stats'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Pressure error'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale = -1
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Temperature error'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale =  1
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Spec hum error'
        sec4_data(index)%nbits =  9
        sec4_data(index)%scale =  5
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'First order stats'
        sec4_data(index)%nbits =  6
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

        index = index + 1
        sec4_data(index)%name  = 'Percent confidence'
        sec4_data(index)%nbits =  7
        sec4_data(index)%scale =  0
        sec4_data(index)%shift =  0

      ENDDO

    ENDIF

! Lev2c
! -----

    index = index + 1
    sec4_data(index)%name  = 'Vert sig'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Sfc geop height'
    sec4_data(index)%nbits = 17
    sec4_data(index)%scale =  0
    sec4_data(index)%shift = -1000

    index = index + 1
    sec4_data(index)%name  = 'Sfc press'
    sec4_data(index)%nbits = 14
    sec4_data(index)%scale = -1
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'First order stats'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Sfc press error'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale = -1
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'First order stats'
    sec4_data(index)%nbits =  6
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0

    index = index + 1
    sec4_data(index)%name  = 'Percent confidence'
    sec4_data(index)%nbits =  7
    sec4_data(index)%scale =  0
    sec4_data(index)%shift =  0


  END SUBROUTINE BUFR_sec4_init

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE BUFR_sec4_to_ROPP(ro_data, ival, rval)

    TYPE(ROprof), INTENT(inout)  :: ro_data
    INTEGER, INTENT(inout)       :: ival(:)
    REAL(dp), INTENT(inout)      :: rval(:)

    CHARACTER (LEN=20)           :: sw_no ! software version
    REAL(dp), PARAMETER          :: ropp_mdtv=-9999.0_dp
    INTEGER                      :: k, l, n1b, n2a, n2b, nfq, index
    REAL(dp)                     :: freq, impact, bangle, sigma, dum

! LEO ID
    SELECT CASE(ival(1))
      CASE (3)
        ro_data%leo_id(1:4) = 'METB'
      CASE (4)
        ro_data%leo_id(1:4) = 'META'
      CASE (5)
        ro_data%leo_id(1:4) = 'METC'
      CASE (40)
        ro_data%leo_id(1:4) = 'OERS'
      CASE (41)
        ro_data%leo_id(1:4) = 'CHMP'
      CASE (42)
        ro_data%leo_id(1:4) = 'TSRX'
      CASE (43)
        ro_data%leo_id(1:4) = 'TDMX'
      CASE (44)
        ro_data%leo_id(1:4) = 'PAZE'
      CASE (66)
        ro_data%leo_id(1:4) = 'SE6A'
      CASE (67)
        ro_data%leo_id(1:4) = 'SE6B'
      CASE (265)
        ro_data%leo_id(1:4) = 'GOP1'
      CASE (266)
        ro_data%leo_id(1:4) = 'GOP2'
      CASE (267)
        ro_data%leo_id(1:4) = 'PLIA'
      CASE (268)
        ro_data%leo_id(1:4) = 'PLIB'
      CASE (269)
        ro_data%leo_id(1:4) = 'SP3U'
      CASE (421)
        ro_data%leo_id(1:4) = 'OSAT'
      CASE (440)
        ro_data%leo_id(1:4) = 'MGTP'
      CASE (522)
        ro_data%leo_id(1:4) = 'FY3C'
      CASE (523)
        ro_data%leo_id(1:4) = 'FY3D'
      CASE (722)
        ro_data%leo_id(1:4) = 'GRAA'
      CASE (723)
        ro_data%leo_id(1:4) = 'GRAB'
      CASE (740, 741, 742, 743, 744, 745)
        WRITE(ro_data%leo_id(1:4), '(A1,I3.3)') 'C', ival(1) - 739
      CASE (750, 751, 752, 753, 754, 755)
        WRITE(ro_data%leo_id(1:4), '(A3,I1.1)') 'C2E', ival(1) - 749
      CASE (786)
        ro_data%leo_id(1:4) = 'CNOF'
      CASE (800)
        ro_data%leo_id(1:4) = 'SUNS'
      CASE (803)
        ro_data%leo_id(1:4) = 'GRAC'
      CASE (804)
        ro_data%leo_id(1:4) = 'GRAD'
      CASE (820)
        ro_data%leo_id(1:4) = 'SACC'
      CASE (825)
        ro_data%leo_id(1:4) = 'KOM5'
    END SELECT

! Software ID
    WRITE ( sw_no, FMT="(F10.3)" ) ival(5) / 1000.D0
    IF ( (ival(5) / 1000.D0) < 10.D0 ) THEN
      ro_data%software_version = "V0" // ADJUSTL ( sw_no )
    ELSE
      ro_data%software_version = "V" // ADJUSTL ( sw_no )
    END IF

! DTocc
    ro_data%dtocc%year   = ival(7)
    ro_data%dtocc%month  = ival(8)
    ro_data%dtocc%day    = ival(9)
    ro_data%dtocc%hour   = ival(10)
    ro_data%dtocc%minute = ival(11)
    ro_data%dtocc%second = INT(rval(12))
    ro_data%dtocc%msec   = NINT((rval(12) - ro_data%dtocc%second)*1000.D0)

! GNSS ID
    SELECT CASE(ival(21))
      CASE (401)
        WRITE(ro_data%gns_id(1:4), '(A1,I3.3)') 'G', ival(22)
      CASE (402)
        WRITE(ro_data%gns_id(1:4), '(A1,I3.3)') 'R', ival(22)
      CASE (403)
        WRITE(ro_data%gns_id(1:4), '(A1,I3.3)') 'E', ival(22)
      CASE (404)
        WRITE(ro_data%gns_id(1:4), '(A1,I3.3)') 'C', ival(22)
    END SELECT

! OCC ID
    SELECT CASE(ival(3))
      CASE (94)
        ro_data%processing_centre = 'DMI (ROM SAF)'
      CASE (78)
        ro_data%processing_centre = 'GFZ Helmholtz Centre, Potsdam'
      CASE (74)
        ro_data%processing_centre = 'METO Met Office, Exeter'
      CASE (60)
        ro_data%processing_centre = 'UCAR Boulder'
      CASE (160)
        ro_data%processing_centre = 'NESDIS Washington'
      CASE (254)
        ro_data%processing_centre = 'EUMETSAT Darmstadt'
      CASE (38)
        ro_data%processing_centre = 'CMA Beijing'
      CASE (28)
        ro_data%processing_centre = 'ISRO New Delhi'
      CASE (178)
        ro_data%processing_centre = 'SPIRE Spire Global, Inc.'
      CASE (179)
        ro_data%processing_centre = 'GEOPTICS GeoOptics, Inc.'
      CASE (180)
        ro_data%processing_centre = 'PLANETIQ PlanetiQ'
    END SELECT
    WRITE(ro_data%occ_id(1:33), '(A3,I4.4,5(I2.2),A1,A4,A1,A4,A1,A4)') &
      'OC_', ival(7), ival(8), ival(9), ival(10), ival(11), INT(rval(12)), '_', &
      ro_data%leo_id(1:4), '_', ro_data%gns_id(1:4), '_', ro_data%processing_centre(1:4)

! PCD
    DO k=0, 15
      IF (IBITS(ival(13), k, 1) == 1) THEN
        ro_data%pcd = IBSET(ro_data%pcd, 15-k)
      ELSE
        ro_data%pcd = IBCLR(ro_data%pcd, 15-k)
      ENDIF
    ENDDO

! Overall quality
  IF (ival(14) == -999) THEN
    ro_data%overall_qual = -9.9999E7_dp
  ELSE
    ro_data%overall_qual = REAL(ival(14), KIND=KIND(1.D0))
  ENDIF

! LEO ref POD
    ro_data%georef%leo_pod%pos(1) = rval(15)
    ro_data%georef%leo_pod%pos(2) = rval(16)
    ro_data%georef%leo_pod%pos(3) = rval(17)
    ro_data%georef%leo_pod%vel(1) = rval(18)
    ro_data%georef%leo_pod%vel(2) = rval(19)
    ro_data%georef%leo_pod%vel(3) = rval(20)

! GNSS ref POD
    ro_data%georef%gns_pod%pos(1) = rval(23)
    ro_data%georef%gns_pod%pos(2) = rval(24)
    ro_data%georef%gns_pod%pos(3) = rval(25)
    ro_data%georef%gns_pod%vel(1) = rval(26)
    ro_data%georef%gns_pod%vel(2) = rval(27)
    ro_data%georef%gns_pod%vel(3) = rval(28)

! Georef
    ro_data%georef%time_offset = rval(29)
    ro_data%georef%lat         = rval(30)
    ro_data%georef%lon         = rval(31)
    ro_data%georef%r_coc(1)    = rval(32)
    ro_data%georef%r_coc(2)    = rval(33)
    ro_data%georef%r_coc(3)    = rval(34)
    ro_data%georef%roc         = rval(35)
    ro_data%georef%azimuth     = rval(36)
    ro_data%georef%undulation  = rval(37)

! Lev1b
    n1b = ival(38)
    index = 38
    DO k=1,n1b
      index = index + 1  ;  ro_data%lev1b%lat_tp(k)     = rval(index)
      index = index + 1  ;  ro_data%lev1b%lon_tp(k)     = rval(index)
      index = index + 1  ;  ro_data%lev1b%azimuth_tp(k) = rval(index)
      index = index + 1  ;  nfq                         = ival(index)

      DO l=1,nfq

        index = index + 1  ;  freq   = rval(index)
        index = index + 1  ;  impact = rval(index)
        index = index + 1  ;  bangle = rval(index)
        index = index + 1  ;  dum    = ival(index) ! First order stats
        index = index + 1  ;  sigma  = rval(index)
        index = index + 1  ;  dum    = ival(index) ! First order stats

        SELECT CASE(NINT(freq/1.0D6))
          CASE (1500)
            ro_data%lev1b%impact_L1(k)       = impact
            ro_data%lev1b%bangle_L1(k)       = bangle
            ro_data%lev1b%bangle_L1_sigma(k) = sigma
          CASE (1200)
            ro_data%lev1b%impact_L2(k)       = impact
            ro_data%lev1b%bangle_L2(k)       = bangle
            ro_data%lev1b%bangle_L2_sigma(k) = sigma
          CASE (0)
            ro_data%lev1b%impact(k)          = impact
            ro_data%lev1b%bangle(k)          = bangle
            ro_data%lev1b%bangle_sigma(k)    = sigma
        END SELECT

      ENDDO

      index = index + 1  ;  dum = ival(index) ! Percent confidence
      IF (dum == -999) THEN
        ro_data%lev1b%bangle_qual(k) = -9.9999E7_dp
      ELSE
        ro_data%lev1b%bangle_qual(k) = REAL(dum, KIND=KIND(1.D0))
      ENDIF

    ENDDO

! Lev2a
    index = index + 1  ;  n2a = ival(index)
    DO k=1,n2a
      index = index + 1  ;  ro_data%lev2a%alt_refrac(k)   = ival(index)
      index = index + 1  ;  ro_data%lev2a%refrac(k)       = rval(index)
      index = index + 1  ;  dum                           = ival(index) ! First order stats
      index = index + 1  ;  ro_data%lev2a%refrac_sigma(k) = rval(index)
      index = index + 1  ;  dum                           = ival(index) ! First order stats
      index = index + 1  ;  dum                           = ival(index) ! Percent confidence
      IF (dum == -999) THEN
        ro_data%lev2a%refrac_qual(k) = -9.9999E7_dp
      ELSE
        ro_data%lev2a%refrac_qual(k) = REAL(dum, KIND=KIND(1.D0))
      ENDIF
! calculate geop_refrac, as it is not present in BUFR file
      ro_data%lev2a%geop_refrac(k) = &
        geometric2geopotential(ro_data%georef%lat, ro_data%lev2a%alt_refrac(k))
    ENDDO

! Lev2b
    index = index + 1  ;  n2b = ival(index)
    DO k=1,n2b
      index = index + 1  ;  ro_data%lev2b%geop(k)        = ival(index)
      index = index + 1  ;  ro_data%lev2b%press(k)       = rval(index)
      IF ( ro_data%lev2b%press(k) > ropp_mdtv ) ro_data%lev2b%press(k) = &
                                                ro_data%lev2b%press(k) / 100.0D0
      index = index + 1  ;  ro_data%lev2b%temp(k)        = rval(index)
      index = index + 1  ;  ro_data%lev2b%shum(k)        = rval(index)
      IF ( ro_data%lev2b%shum(k) > ropp_mdtv ) ro_data%lev2b%shum(k) = &
                                               ro_data%lev2b%shum(k) * 1000.0D0
      index = index + 1  ;  dum                          = ival(index) ! First order stats
      index = index + 1  ;  ro_data%lev2b%press_sigma(k) = rval(index)
      IF ( ro_data%lev2b%press_sigma(k) > ropp_mdtv ) ro_data%lev2b%press_sigma(k) = &
                                                      ro_data%lev2b%press_sigma(k) / 100.0D0
      index = index + 1  ;  ro_data%lev2b%temp_sigma(k)  = rval(index)
      index = index + 1  ;  ro_data%lev2b%shum_sigma(k)  = rval(index)
      IF ( ro_data%lev2b%shum_sigma(k) > ropp_mdtv ) ro_data%lev2b%shum_sigma(k) = &
                                                     ro_data%lev2b%shum_sigma(k) * 1000.0D0
      index = index + 1  ;  dum                          = ival(index) ! First order stats
      index = index + 1  ;  dum                          = ival(index) ! Percent confidence
      IF (dum == -999) THEN
        ro_data%lev2b%meteo_qual(k) = -9.9999E7_dp
      ELSE
        ro_data%lev2b%meteo_qual(k) = REAL(dum, KIND=KIND(1.D0))
      ENDIF
    ENDDO

! Lev2c
    index = index + 1  ;  dum                           = ival(index) ! Vertical significance
    index = index + 1  ;  ro_data%lev2c%geop_sfc        = ival(index)
    index = index + 1  ;  ro_data%lev2c%press_sfc       = rval(index)
    IF ( ro_data%lev2c%press_sfc > ropp_mdtv ) ro_data%lev2c%press_sfc = &
                                               ro_data%lev2c%press_sfc / 100.0D0
    index = index + 1  ;  dum                           = ival(index) ! First order stats
    index = index + 1  ;  ro_data%lev2c%press_sfc_sigma = rval(index)
    IF ( ro_data%lev2c%press_sfc_sigma > ropp_mdtv ) ro_data%lev2c%press_sfc_sigma = &
                                                     ro_data%lev2c%press_sfc_sigma / 100.0D0
    index = index + 1  ;  dum                           = ival(index) ! First order stats
    index = index + 1  ;  ro_data%lev2c%press_sfc_qual  = ival(index) ! Percent confidence

  END SUBROUTINE BUFR_sec4_to_ROPP

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE BUFR_decode_sample(cdata, namvar, i1, i2, pad, size, quiet)

    CHARACTER(LEN=*), INTENT(IN)       :: cdata, namvar
    INTEGER, INTENT(IN)                :: i1, i2
    LOGICAL, OPTIONAL, INTENT(INOUT)   :: pad
    LOGICAL, OPTIONAL, INTENT(IN)      :: quiet
    INTEGER, OPTIONAL, INTENT(INOUT)   :: size

    INTEGER                            :: i
    CHARACTER(LEN=10)                  :: c_dum
    INTEGER                            :: dum

    CHARACTER(LEN=5)                   :: fmt
    CHARACTER(LEN=5)                   :: s_octet, s_val
    CHARACTER(LEN=128)                 :: msg
    LOGICAL                            :: local_quiet

    local_quiet = .FALSE.
    IF ( PRESENT(quiet) ) local_quiet = quiet

    DO i=i1, i2
      WRITE(c_dum(2*(i-i1)+1:2*(i-i1)+2), '(Z2)') ICHAR(cdata(i:i))
    ENDDO

    IF (i2 - i1 >= 4) THEN
      WRITE(fmt, '(A,I2,A)') '(Z', 2*(i2-i1+1), ')'
    ELSE
      WRITE(fmt, '(A,I1,A)') '(Z', 2*(i2-i1+1), ')'
    ENDIF

    READ(c_dum, fmt) dum

! Extra diags, for debugging.
!    PRINT*, ' '
!    PRINT*, TRIM(ADJUSTL(namvar)) // ' = ', dum
!    DO i=i1, i2
!      PRINT*,'Byte ', i
!      CALL BUFR_bit_breakdown(cdata(i:i))
!    ENDDO

    IF ( .NOT. local_quiet ) THEN
      WRITE(s_octet, '(I2,A1,I2)') i1, '-', i2
      WRITE(s_val, '(I5)') dum
      WRITE(msg, '(2X,A5,3X,A24,3X,A5)') s_octet, TRIM(namvar), s_val
      IF ( s_octet(1:1) == ' ' ) THEN
        CALL message(msg_noin, '   ' // ADJUSTL(msg))
      ELSE
        CALL message(msg_noin, '  ' // ADJUSTL(msg))
      ENDIF
    ENDIF

    IF ( PRESENT(pad) .AND. dum == 10 ) pad = .TRUE.

    IF ( PRESENT(size) ) size = dum

  END SUBROUTINE BUFR_decode_sample

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE BUFR_bit_breakdown(cchar)

    CHARACTER(LEN=1)                   :: cchar

    CHARACTER(LEN=8)                   :: ctemp
!    CHARACTER(LEN=10)                  :: cdum
    INTEGER                            :: m, j

!    PRINT*,'cchar = ', cchar

    m = ICHAR(cchar)
!    WRITE(cdum, '(Z2)') cchar
!    READ(cdum, '(Z2)') m
!    PRINT*,'ichar(cchar) = ', m

    ctemp = '00000000'
    DO j=0,7
      IF (BTEST(m, j)) ctemp(j+1:j+1) = '1'
    ENDDO

    PRINT*,'Bit breakdown: ', ctemp

  END SUBROUTINE BUFR_bit_breakdown

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  FUNCTION ival(cstring) RESULT(ivalue)

! There must be a simpler way of doing this!

    CHARACTER(LEN=*) :: cstring
    INTEGER          :: ivalue

    INTEGER          :: k, l, m

    ivalue = 0
    m = 1
    DO k=LEN(cstring),1,-1
      READ(cstring(k:k), '(I1)') l
      ivalue = ivalue + l*m
      m = 2 * m
    ENDDO

  END FUNCTION ival

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

END PROGRAM robufr2ropp
