!ifort -o azi.exe azi.f90
!
PROGRAM azi

  IMPLICIT none

  INTEGER, PARAMETER                      :: wp=KIND(1.0D0)
  REAL(KIND=wp), PARAMETER                :: pi=4.d0*ATAN(1.d0), dtor=pi/180.d0, rad2deg=1.0d0/dtor
  REAL(KIND=wp), PARAMETER                :: r_earth=6.4d6

  INTEGER, PARAMETER                      :: npts=1
  INTEGER                                 :: i

  REAL(KIND=wp), DIMENSION(npts,3)        :: r_leo                      ! in ECF coords
  REAL(KIND=wp), DIMENSION(npts)          :: rad_leo, lon_leo, lat_leo  ! in ECF coords

  REAL(KIND=wp), DIMENSION(npts,3)        :: r_gns                      ! in ECF coords
  REAL(KIND=wp), DIMENSION(npts)          :: rad_gns, lon_gns, lat_gns  ! in ECF coords

  REAL(KIND=wp), DIMENSION(3), PARAMETER  :: pa=(/ 0.0d0, 0.0d0, 1.0d0 /)
  REAL(KIND=wp), DIMENSION(npts,3)        :: perigee
  REAL(KIND=wp)                           :: slta, ro, alpha, theta
  REAL(KIND=wp), DIMENSION(npts)          :: azimuth_tp, azimuth_ropp, azimuth_gpac

  CHARACTER(LEN=256)                      :: title

! 1.1 Define satellite positions (ECF coordinates)
! ------------------------------------------------

  DO i=1,npts
  
!    title = 'LEO due west of GNS'
!    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) =   0.0d0 * dtor ; lon_leo(i) = -10.d0 * dtor
!    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) =   0.0d0 * dtor ; lon_gns(i) =  90.d0 * dtor

!    title = 'LEO due east of GNS'
!    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) =   0.0d0 * dtor ; lon_leo(i) =  10.d0 * dtor
!    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) =   0.0d0 * dtor ; lon_gns(i) = -90.d0 * dtor

    title = 'LEO due north of GNS'
    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) =  10.0d0 * dtor ; lon_leo(i) =   0.d0 * dtor
    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) = -10.0d0 * dtor ; lon_gns(i) =   0.d0 * dtor

!    title = 'LEO due south of GNS'
!    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) = -10.0d0 * dtor ; lon_leo(i) =   0.d0 * dtor
!    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) =  10.0d0 * dtor ; lon_gns(i) =   0.d0 * dtor

!    title = 'LEO and GEO as in 1st picture of #233'
!    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) =  55.0d0 * dtor ; lon_leo(i) = -25.d0 * dtor
!    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) = -50.0d0 * dtor ; lon_gns(i) =   0.d0 * dtor

!    title = 'LEO and GEO as in 2nd picture of #233'
!    rad_leo(i) = r_earth +   800.0d3 ; lat_leo(i) =  60.0d0 * dtor ; lon_leo(i) =  -40.d0 * dtor
!    rad_gns(i) = r_earth + 20000.0d3 ; lat_gns(i) =   5.0d0 * dtor ; lon_gns(i) = -170.d0 * dtor


    r_leo(i,:) = rad_leo(i) * (/ COS(lon_leo(i))*COS(lat_leo(i)), SIN(lon_leo(i))*COS(lat_leo(i)), SIN(lat_leo(i)) /)

    r_gns(i,:) = rad_gns(i) * (/ COS(lon_gns(i))*COS(lat_gns(i)), SIN(lon_gns(i))*COS(lat_gns(i)), SIN(lat_gns(i)) /)


! 1.2 Determine ray tangent points
! --------------------------------
     
    slta = impact_parameter(r_leo(i,:), r_gns(i,:))
    ro = SQRT(DOT_PRODUCT(r_leo(i,:), r_leo(i,:)))   
    alpha = ACOS(slta/ro)

    perigee(i,:) = rotate(r_leo(i,:), vector_product(r_leo(i,:), r_gns(i,:)), alpha) * (slta/ro)

    PRINT*,'slta,ro,alpha,perigee = ', slta,ro,alpha,perigee(i,:)

  ENDDO


! 1.4 Cross-section azimuth at tangent point as in ROPP
! -----------------------------------------------------

  DO i=1,size(r_leo,1)

    theta = vector_angle(vector_product(r_gns(i,:),r_leo(i,:)), &
                         vector_product(pa, perigee(i,:)))

    PRINT*,'theta_ropp = ', theta

    PRINT*,'DOT_PRODUCT(r_gns(i,:)-r_leo(i,:),vector_product(pa, perigee(i,:))) = ', &
            DOT_PRODUCT(r_gns(i,:)-r_leo(i,:),vector_product(pa, perigee(i,:)))

    if (DOT_PRODUCT(r_gns(i,:)-r_leo(i,:),vector_product(pa, perigee(i,:))) < 0) THEN
      theta = 2.0d0*Pi - theta
    endif

    PRINT*,'theta_ropp = ', theta

    azimuth_tp(i) = theta * rad2deg

    azimuth_ropp(i) = azimuth_tp(i)

  ENDDO

! 1.4 Cross-section azimuth at tangent point as in GPAC
! -----------------------------------------------------

  DO i=1,size(r_leo,1)

    theta = vector_angle(vector_product(perigee(i,:), pa), &
                         vector_product(r_gns(i,:),r_leo(i,:)), -perigee(i,:))

    PRINT*,'theta_gpac = ', theta

    azimuth_tp(i) = theta * rad2deg

    PRINT*,'azimuth_gpac = ', azimuth_tp(i)

    if (azimuth_tp(i) < 0.0_wp ) azimuth_tp(i) = azimuth_tp(i) + 360.0_wp

    PRINT*,'azimuth_gpac = ', azimuth_tp(i)

    azimuth_gpac(i) = azimuth_tp(i)

  ENDDO


PRINT*, '*** ' // TRIM(title) // ' ***'
PRINT*, 'azimuth_ropp = ', azimuth_ropp
PRINT*, 'azimuth_gpac = ', azimuth_gpac



CONTAINS

!****f* Coordinates/rotate
!
! NAME
!    rotate - Rotate a vector in cartesian coordinates around
!             a given axis by a given angle  
!
! SYNOPSIS
!    Rotate = rotate(X, A, phi)
! 
! DESCRIPTION
!    This function rotates a vector X around a given axis A by angle phi.
!       N*(N,X) + [N,X]*Sin(Phi) + (X-N*(N,X))*Cos(Phi),   where N=A/|A|
!
! INPUTS
!    X             Vector to rotate
!    A             Rotation axis
!    Phi           Rotation angle (rad)
!
! OUTPUT
!    Rotate        Rotated vector
!
! NOTES
!
! SEE ALSO
!
! REFERENCES
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the ROM SAF
!   Helpdesk at http://www.romsaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

function rotate(X, A, Phi) result(R)

! 1.1 Declarations
! ----------------

  implicit none

  real(wp), dimension(3), intent(in) :: X        ! input vector
  real(wp), dimension(3), intent(in) :: A        ! rotation axis
  real(wp),               intent(in) :: phi      ! rotation angle
  real(wp), dimension(3)             :: R        ! rotated vector

  real(wp), dimension(3) :: norm         ! normed rotation axis

! 1.2 Frame rotation
! ------------------

!   N*(N,X) + [N,X]*Sin(Phi) + (X-N*(N,X))*Cos(Phi),   where N=A/|A|

  norm = A(:)/Sqrt(Dot_Product(A(:), A(:)))
 
  R = norm*(Dot_Product(norm, X)) + vector_product(norm, X)*sin(phi)  &
       + (X - norm*Dot_Product(norm,X))*cos(phi)

end function rotate

!****f* Coordinates/vector_product
!
! NAME
!    vector_product - Compute a vector product of two cartesian vectors
!
! SYNOPSIS
!    product = vector_product(X, Y)
!
! INPUTS
!    X             Vector 1
!    Y             Vector 2
!
! OUTPUT
!    Product       Vector product
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the ROM SAF
!   Helpdesk at http://www.romsaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****

  function vector_product(X, Y) result(product)
    
    real(wp), dimension(3), intent(in) :: X
    real(wp), dimension(3), intent(in) :: Y
    real(wp), dimension(3)             :: product
    
    product = (/ X(2)*Y(3) - X(3)*Y(2),  &
                 X(3)*Y(1) - X(1)*Y(3),  &
                 X(1)*Y(2) - X(2)*Y(1) /)
    
  end function vector_product

!****f* Coordinates/vector_angle
!
! NAME
!    vector_angle - Find the angle between two cartesian vectors
!
! SYNOPSIS
!    angle = vector_angle(X, Y, A)
!
! INPUTS
!    X             Vector 1
!    Y             Vector 2
!    A             Orientation axis (optional)
!
! OUTPUT
!    Angle       Angle between vectors
!
! AUTHOR
!   Met Office, Exeter, UK.
!   Any comments on this software should be given via the ROM SAF
!   Helpdesk at http://www.romsaf.org
!
! COPYRIGHT
!   (c) EUMETSAT. All rights reserved.
!   For further details please refer to the file COPYRIGHT
!   which you should have received as part of this distribution.
!
!****
  
function vector_angle(X, Y, A) result(angle)
  
  real(wp), dimension(3), intent(in) :: X
  real(wp), dimension(3), intent(in) :: Y
  real(wp), dimension(3), optional, intent(in) :: A
  real(wp)                           :: angle
  
  real(wp), dimension(3) :: n, alpha, beta, gamma
  real(wp)               :: nn
  
  if (present(A)) then
     n = A
  else
     n = vector_product(X, Y)
  endif
  
  nn = Dot_Product(n, n)

  if (nn == 0) then
     angle = 0.0_wp
  else
     n = n/sqrt(nn)
     alpha = vector_product(n, X)
     
     beta = X - Dot_Product(n, X) * n
     gamma = Y - Dot_Product(n, Y) * n
     angle = atan2(Dot_Product(alpha,gamma), Dot_Product(beta,gamma))
  endif
      
end function vector_angle


function impact_parameter(r_leo, r_gns, bangle) result(impact)

! 1.1 Declarations
! ----------------

  implicit none

  real(wp), dimension(3), intent(in) :: r_leo   ! LEO position vector (ECF)
  real(wp), dimension(3), intent(in) :: r_gns   ! GPS position vector (ECF)
  real(wp), optional, intent(in)     :: bangle  ! Bending angle
  real(wp)                           :: impact  ! Impact parameter

  real(wp)               :: r0       ! Length of r_leo
  real(wp)               :: r1       ! Length of r_gns
  real(wp)               :: omega    ! Complementary to r_leo^r_gns - bangle
  real(wp)               :: talpha   ! Tan(r_leo^(r_leo-r_gns))

! 1.2 Length of vectors r_leo and r_gns
! -------------------------------------

  r0 = Sqrt(Dot_Product(r_leo, r_leo))
  r1 = Sqrt(Dot_Product(r_gns, r_gns))

! 1.3 Find vector angle between r_leo and r_gns
! ---------------------------------------------

  omega = Pi - vector_angle(r_leo, r_gns)

  if (present(bangle)) then
     omega = omega + bangle
  endif

! 1.4 Determine impact parameter by trigonometry
! ----------------------------------------------

  talpha = r1*Sin(omega) / (r0 + r1*Cos(omega))

  impact = r0*talpha / Sqrt(1.0_wp + talpha**2)

end function impact_parameter




end program azi









