!export LIBDIR=/project/ukmo/rhel6/netcdf4/ifort_composerxe
!ifort -o und_dat2nc.exe und_dat2nc.f90 -I$LIBDIR/include -L$LIBDIR/lib -lnetcdf -lnetcdff

PROGRAM und_dat2nc

!
! Convert ropp_pp/data/egm96.dat and ropp_pp/data/corrcoef.dat into a
! single netcdf file, which need not be zipped.
!
! IDC, 23/12/2016
!

  USE netcdf

  IMPLICIT NONE

  INTEGER, PARAMETER          :: wp=KIND(1.D0)

  INTEGER, PARAMETER          :: ll=65341  ! = 361 * 362 / 2, i.e. for a 1/4 deg grid
                                           ! (cos(i*360*lon) ~ (1, 0, -1, 0) per degree of lon)

  CHARACTER(nf90_max_name)    :: ifile1='egm96.dat'
  INTEGER, PARAMETER          :: lun1=11

  CHARACTER(nf90_max_name)    :: ifile2='corrcoef.dat'
  INTEGER, PARAMETER          :: lun2=12

  REAL(wp), DIMENSION(ll)     :: hc, hs, cc, cs, hc2, cc2
  CHARACTER(nf90_max_name)    :: ofile='und_coeffs.nc'

  INTEGER                     :: i, n, m, ierr
  INTEGER                     :: ncid, ll_dimid
  INTEGER                     :: hc_varid, hs_varid, cs_varid, cc_varid
  REAL(wp)                    :: c, s

! 1. Input undulation coefficients
! --------------------------------

  OPEN ( UNIT=lun1, STATUS="OLD", FILE=ifile1, ACTION="READ" )

  hc = 0.0_wp  ;  hs = 0.0_wp

  DO
    READ ( UNIT=lun1, FMT=*, IOSTAT=ierr ) n, m, c, s
    IF ( ierr /= 0 ) EXIT
    i = ( n * ( n + 1 ) ) / 2 + m + 1
    hc(i) = c
    hs(i) = s
  END DO

  CLOSE ( UNIT=lun1 )

  WRITE(*, '(/,A,/,(5E30.20))') 'From ' // TRIM(ADJUSTL(ifile1)) // ': hc(1:10) = ', hc(1:10)

! 2. Input correction coefficients
! --------------------------------

  OPEN ( UNIT=lun2, STATUS="OLD", FILE=ifile2, ACTION="READ" )

  cc = 0.0_wp  ;  cs = 0.0_wp

  DO
    READ ( UNIT=lun2, FMT=*, IOSTAT=ierr ) n, m, c, s
    IF ( ierr /= 0 ) EXIT
    i = ( n * ( n + 1 ) ) / 2 + m + 1
    cc(i) = c
    cs(i) = s
  END DO

  CLOSE ( UNIT=lun2 )

  WRITE(*, '(/,A,/,(5E30.20))') 'From ' // TRIM(ADJUSTL(ifile2)) // ': cc(1:10) = ', cc(1:10)

! 3. Output both
! --------------

! Create output file
  CALL check(nf90_create(ofile, nf90_clobber, ncid))

! Define the dimensions
  CALL check(nf90_def_dim(ncid, 'n', ll, ll_dimid))

! Define the variables
  CALL check(nf90_def_var(ncid, 'hc', nf90_double, ll_dimid, hc_varid))
  CALL check(nf90_def_var(ncid, 'hs', nf90_double, ll_dimid, hs_varid))
  CALL check(nf90_def_var(ncid, 'cc', nf90_double, ll_dimid, cc_varid))
  CALL check(nf90_def_var(ncid, 'cs', nf90_double, ll_dimid, cs_varid))

! Define attributes
  CALL check(nf90_put_att(ncid, hc_varid, 'long_name', 'First (cosine) undulation coefficient'))
  CALL check(nf90_put_att(ncid, hc_varid, 'units', '1'))
  CALL check(nf90_put_att(ncid, hs_varid, 'long_name', 'Second (sine) undulation coefficient'))
  CALL check(nf90_put_att(ncid, hs_varid, 'units', '1'))
  CALL check(nf90_put_att(ncid, cc_varid, 'long_name', 'First (cosine) correction coefficient'))
  CALL check(nf90_put_att(ncid, cc_varid, 'units', '1'))
  CALL check(nf90_put_att(ncid, cs_varid, 'long_name', 'Second (sine) correction coefficient'))
  CALL check(nf90_put_att(ncid, cs_varid, 'units', '1'))
  CALL check(nf90_put_att(ncid, nf90_global, 'Source', 'Derived from WGS 84 EGM96 15-Minute geoid height and correction coefficients'))
  CALL check(nf90_put_att(ncid, nf90_global, 'Reference', 'http://earth-info.nga.mil/GandG/wgs84/gravitymod/egm96/egm96.html'))
  CALL check(nf90_put_att(ncid, nf90_global, 'Note', 'In early versions of ROPP, these data were held in ropp_pp/data/egm96.dat and ropp_pp/data/corrcoef.dat'))
  CALL check(nf90_put_att(ncid, nf90_global, 'Author', 'ROM SAF, 2017'))

! Come out of definition mode
  CALL check(nf90_enddef(ncid))

! Write out variables
  CALL check(nf90_put_var(ncid, hc_varid, hc))
  CALL check(nf90_put_var(ncid, hs_varid, hs))
  CALL check(nf90_put_var(ncid, cc_varid, cc))
  CALL check(nf90_put_var(ncid, cs_varid, cs))

! Close the output file
  CALL check(nf90_close(ncid))

! Check that we can read it
  hc2 = 0.0_wp
  CALL check(nf90_open(ofile, nf90_nowrite, ncid))
  CALL check(nf90_inq_varid(ncid, 'hc', hc_varid))
  CALL check(nf90_get_var(ncid, hc_varid, hc2))
  CALL check(nf90_close(ncid))

  WRITE(*, '(/,A,/,(5E30.20))') 'From ' // TRIM(ADJUSTL(ofile)) // ': hc(1:10) = ', hc2(1:10)

  WRITE(*, '(/,A,/,(5E30.20))') TRIM(ADJUSTL(ifile1)) // ' - ' // TRIM(ADJUSTL(ofile)) // ': hc(1:10) = ', hc(1:10)-hc2(1:10)

  cc2 = 0.0_wp
  CALL check(nf90_open(ofile, nf90_nowrite, ncid))
  CALL check(nf90_inq_varid(ncid, 'cc', cc_varid))
  CALL check(nf90_get_var(ncid, cc_varid, cc2))
  CALL check(nf90_close(ncid))

  WRITE(*, '(/,A,/,(5E30.20))') 'From ' // TRIM(ADJUSTL(ofile)) // ': cc(1:10) = ', cc2(1:10)

  WRITE(*, '(/,A,/,(5E30.20))') TRIM(ADJUSTL(ifile1)) // ' - ' // TRIM(ADJUSTL(ofile)) // ': cc(1:10) = ', cc(1:10)-cc2(1:10)

CONTAINS


!----------------------------------------------------------------------------

  SUBROUTINE check(status)

    INTEGER, INTENT (IN) :: status

    IF (status /= nf90_noerr) THEN
      PRINT*, TRIM(nf90_strerror(status))
      STOP "STOPPED"
    END IF

  END SUBROUTINE check


END PROGRAM und_dat2nc
